"""LakData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data import data_util
from xms.mf6.data.list_package_data import ListPackageData
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import gui_util, units_util
from xms.mf6.gui.options_defs import Checkbox, CheckboxButton, CheckboxField


class LakData(ListPackageData):
    """Data class to hold the info for the LAK package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'LAK6'
        self.outlets_filename = ''
        self.tables = {}  # dict with key = lake id, value = filename
        self.block_with_cellids = 'CONNECTIONDATA'
        self.list_blocks = {'PACKAGEDATA': '', 'CONNECTIONDATA': '', 'OUTLETS': ''}

    # @overrides
    def get_column_delegate_info(self, block):
        """Returns a list of tuples of [0] column index and [1] list of strings."""
        delegate_info = None

        names, _, _ = self.get_column_info(block)
        if block == 'CONNECTIONDATA':
            delegate_info = [(names.index('CLAKTYPE'), ['VERTICAL', 'HORIZONTAL', 'EMBEDDEDH', 'EMBEDDEDV'])]
        elif block == 'OUTLETS':
            delegate_info = [(names.index('COUTTYPE'), ['SPECIFIED', 'MANNING', 'WEIR'])]
        elif not block or block == 'PERIODS':
            delegate_info = [
                (
                    names.index('LAKSETTING'), [
                        'STATUS', 'STAGE', 'RAINFALL', 'EVAPORATION', 'RUNOFF', 'INFLOW', 'WITHDRAWAL', 'RATE',
                        'INVERT', 'WIDTH', 'SLOPE', 'ROUGH', 'AUXILIARY'
                    ]
                )
            ]

        return delegate_info

    # @overrides
    def get_column_info(self, block, use_aux=True):
        """Returns column names, types, and defaults.

        The columns depend on the DIS package in use and the AUX variables.
        The package specific and AUX columns are type object because they
        might contain time series strings.

        Args:
            block (str): Name of the list block.
            use_aux (bool): True to include AUXILIARY variables.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        if block.upper() == 'PACKAGEDATA':
            columns = {
                'LAKENO': (np.int32, 1),
                'STRT': (np.float64, 0.0),
                'NLAKECONN': (np.int32, 1),
            }
            self.add_aux_columns_to_dict(columns, use_aux=use_aux)
            data_util.add_boundname_columns_to_dict(self.options_block, columns)

        elif block.upper() == 'CONNECTIONDATA':
            id_columns = data_util.get_id_column_dict(self.grid_info())
            columns = {
                **{
                    'LAKENO': (np.int32, 1),
                    'ICONN': (np.int32, 1)
                },
                **id_columns,
                **{
                    'CLAKTYPE': (object, 'VERTICAL'),
                    'BEDLEAK': (np.float64, 0.0),
                    'BELEV': (np.float64, 0.0),
                    'TELEV': (np.float64, 0.0),
                    'CONNLEN': (np.float64, 0.0),
                    'CONNWIDTH': (np.float64, 0.0),
                }
            }

        elif block.upper() == 'OUTLETS':
            columns = {
                'OUTLETNO': (np.int32, 1),
                'LAKEIN': (np.int32, 1),
                'LAKEOUT': (np.int32, 1),
                'COUTTYPE': (object, 'MANNING'),
                'INVERT': (np.float64, 0.0),
                'WIDTH': (np.float64, 0.0),
                'ROUGH': (np.float64, 0.0),
                'SLOPE': (np.float64, 0.0),
            }

        else:  # This would be the stress periods
            columns = {
                'NUMBER': (np.int32, 1),  # integer value that defines the lake or outlet number
                'LAKSETTING': (object, ''),  # Both laksetting and outlet setting
                'VALUE1': (object, ''),
                'VALUE2': (object, ''),
            }

        names, types, defaults = gui_util.column_info_tuple_from_dict(columns)
        return names, types, defaults

    def get_column_tool_tips(self, block: str) -> dict[int, str]:
        """Returns a dict with column index and tool tip.

        Args:
            block (str): Name of the block.
        """
        names, _types, _defaults = self.get_column_info(block)
        if block.upper() == 'PACKAGEDATA':
            length_units = units_util.string_from_units(self, units_util.UNITS_LENGTH)
            return {
                names.index('LAKENO'): 'Lake number',
                names.index('STRT'): f'Starting stage for the lake {length_units}',
                names.index('NLAKECONN'): 'Number of GWF cells connected to this lake. Must be greater than zero.',
            }
        elif block.upper() == 'CONNECTIONDATA':
            length_units = units_util.string_from_units(self, units_util.UNITS_LENGTH)
            return {
                names.index('LAKENO'): 'Lake number',
                names.index('ICONN'): 'GWF connection number for this lake connection entry',
                names.index('CLAKTYPE'): 'Character string that defines the lake-GWF connection type',
                names.index('BEDLEAK'): 'Bed leakance for the lake-GWF connection',
                names.index('BELEV'): f'Bottom elevation for a HORIZONTAL lake-GWF connection {length_units}',
                names.index('TELEV'): f'Top elevation for a HORIZONTAL lake-GWF connection {length_units}',
                names.index('CONNLEN'):
                    'Distance between the connected GWF CELLID node and the lake for a'
                    f' HORIZONTAL, EMBEDDEDH, or EMBEDDEDV lake-GWF connection {length_units}',
                names.index('CONNWIDTH'):
                    f'Connection face width for a HORIZONTAL lake-GWF connection'
                    f' {length_units}',
            }
        elif block.upper() == 'OUTLETS':
            length_units = units_util.string_from_units(self, units_util.UNITS_LENGTH)
            return {
                names.index('OUTLETNO'): 'Outlet number',
                names.index('LAKEIN'): 'Lake number that outlet is connected to',
                names.index('LAKEOUT'): 'Lake number that outlet discharge from lake outlet OUTLETNO is routed to',
                names.index('COUTTYPE'): 'Character string that defines the outlet type',
                names.index('INVERT'): f'Invert elevation for the lake outlet {length_units}',
                names.index('WIDTH'): f'Width of the lake outlet {length_units}',
                names.index('ROUGH'): 'Roughness coefficient for the lake outlet',
                names.index('SLOPE'): 'Bed slope for the lake outlet',
            }
        else:  # stress periods
            names, _types, _defaults = self.get_column_info('')
            return {
                names.index('NUMBER'): 'Lake or outlet number',
                names.index('LAKSETTING'): 'Keyword to start lake and outlet setting line',
            }

    def package_column_info(self, block=''):
        """Returns the column info just for the columns unique to this package.

        You should override this method.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        raise NotImplementedError()
        # return [], {}, {}

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Lake (LAK) Package'

    @override
    def get_time_series_columns(self) -> list[int]:
        """Returns a list of the column indices that can contain time series.

        Returns:
            List of indices of columns that can contain time series.
        """
        names, _, _ = self.get_column_info('PERIODS')
        ts_columns = {'VALUE1', 'VALUE2'}
        return [i for i, name in enumerate(names) if name in ts_columns]

    def block_with_aux(self):
        """Returns the name of the block that can have aux variables.

        Returns:
            (str): The name of the block that can have aux variables.
        """
        return 'PACKAGEDATA'

    def block_with_boundnames(self):
        """Returns the name of the block that can have aux variables.

        Returns:
            (str): The name of the block that can have aux variables.
        """
        return 'PACKAGEDATA'

    # @overrides
    def update_displayed_cell_indices(self) -> None:
        """Updates the cell indices file used to display symbols."""
        self._update_displayed_cell_indices_in_block('CONNECTIONDATA')

    def stress_id_columns(self):
        """Returns the column name where the id exists that can be used to help identify this stress across periods.

        Typically is 'CELLIDX' which is added by GMS but is 'RNO' for SFR.

        Returns:
            See description.
        """
        return ['NUMBER']

    def plottable_columns(self):
        """Returns a set of columns (0-based) that can be plotted with the XySeriesEditor.

        Returns:
            See description.
        """
        column_count = len(self.get_column_info('')[0])
        return set(range(2, column_count))  # Start after the LAKSETTING column

    # @overrides
    def _setup_options(self):
        """Returns the definition of all the available options.

        Returns:
            (OptionsBlock): See description.
        """
        return OptionsBlock(
            [
                CheckboxButton(
                    'AUXILIARY',
                    button_text='Auxiliary Variables...',
                    check_box_method='on_chk_auxiliary',
                    button_method='on_btn_auxiliary'
                ),
                Checkbox('BOUNDNAMES', brief='Allow boundary names', check_box_method='on_chk_boundnames'),
                Checkbox('PRINT_INPUT', brief='Print input to listing file'),
                Checkbox('PRINT_STAGE', brief='Print stage to listing file'),
                Checkbox('PRINT_FLOWS', brief='Print flows to listing file'),
                Checkbox('SAVE_FLOWS', brief='Save flows to budget file'),
                CheckboxField('STAGE FILEOUT', brief='Save stage to file', type_='str'),
                CheckboxField('BUDGET FILEOUT', brief='Save budget to file', type_='str'),
                CheckboxField('PACKAGE_CONVERGENCE FILEOUT', brief='Save convergence info to file', type_='str'),
                CheckboxButton(
                    'TS6 FILEIN', brief='Time-series files', button_text='Files...', button_method='on_btn_ts6_filein'
                ),
                CheckboxButton(
                    'OBS6 FILEIN',
                    brief='Observation files',
                    button_text='Files...',
                    button_method='on_btn_obs6_filein'
                ),
                Checkbox('MOVER', brief='Can be used with the Water Mover (MVR) Package'),
                CheckboxField(
                    'SURFDEP',
                    brief='Surface depression depth for VERTICAL lake-GWF connections',
                    type_='float',
                    value=0.0
                ),
                CheckboxField(
                    'MAXIMUM_ITERATIONS',
                    brief='Maximum number of Newton-Raphson iterations allowed for a lake',
                    type_='int',
                    value=100,
                    minimum=0,
                    maximum=1e6
                ),
                CheckboxField('MAXIMUM_STAGE_CHANGE', brief='Lake stage closure tolerance', type_='float', value=1e-5),
                CheckboxField(
                    'TIME_CONVERSION', brief='Conversion value for outlet flow terms', type_='float', value=1.0
                ),
                CheckboxField(
                    'LENGTH_CONVERSION', brief='Conversion value for outlet flow terms', type_='float', value=1.0
                ),
            ]
        )
