"""NpfData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.griddata_base import GriddataBase
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import options_util, units_util
from xms.mf6.gui.options_defs import Checkbox, Checkbox3Fields, CheckboxCheckbox, CheckboxComboBox, CheckboxField
from xms.mf6.mapping.map_util import Mapping


class NpfData(GriddataBase):
    """Data class to hold the info from a NPF package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'NPF6'
        self.add_block('GRIDDATA', ['ICELLTYPE', 'K', 'K22', 'K33', 'ANGLE1', 'ANGLE2', 'ANGLE3', 'WETDRY'])

    def get_units(self, array_name: str) -> str:
        """Returns a units string.

        Args:
            array_name (str): The name of a array.

        Returns:
            (str): The units string like 'L' or 'L^3/T'.
        """
        units_str = ''
        match array_name:
            case 'ICELLTYPE':
                units_str = units_util.UNITS_UNITLESS
            case 'K' | 'K22' | 'K33':
                units_str = units_util.UNITS_K
            case 'ANGLE1' | 'ANGLE2' | 'ANGLE3':
                units_str = units_util.UNITS_DEGREES
            case 'WETDRY':
                units_str = units_util.UNITS_UNITLESS
            case _:
                return ''  # This is an error
        return units_str

    def is_int_array(self, array_name):
        """Returns True if the array is integers.

        Args:
            array_name (str): The name of a array.

        Returns:
            (bool): True or False
        """
        return array_name.upper() == 'ICELLTYPE'

    @override
    def is_required_array(self, array_name: str) -> bool:
        """Returns True if the array is required.

        Args:
            array_name (str): The name of a array.

        Returns:
            (bool): True or False
        """
        return array_name.upper() in {'ICELLTYPE', 'K'}

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Node Property Flow (NPF) Package'

    def map_info(self, feature_type):
        """Returns info needed for Map from Coverage.

        Args:
            feature_type (str): 'points', 'arcs', or 'polygons'

        Returns:
            (dict): Dict describing how to get the MODFLOW variable from the shapefile or att table fields.
        """
        if feature_type != 'polygons':
            return {}
        return {
            'HK': Mapping(mf_name='K'),
            'VK': Mapping(mf_name='K33'),
            'HANI': Mapping(mf_name='K22'),
            'VANI': Mapping(mf_name='K33')
        }

    def map_import_info(self, feature_type):
        """Returns info needed for mapping shapefile or transient data file to package data.

        Args:
            feature_type (str): 'points', 'arcs', or 'polygons'

        Returns:
            (dict): See description
        """
        return {}

    @override
    def _setup_options(self) -> OptionsBlock:
        """Returns the definition of all the available options.

        Returns:
            See description.
        """
        return OptionsBlock(
            [
                Checkbox('SAVE_FLOWS', brief='Save flows to budget file'),
                Checkbox('PRINT_FLOWS', brief='Print flows between cells to listing file'),
                CheckboxComboBox(
                    'ALTERNATIVE_CELL_AVERAGING',
                    brief='Alternative conductance calculation method',
                    items=['LOGARITHMIC', 'AMT-LMK', 'AMT-HMK'],
                    value='AMT-HMK'
                ),
                Checkbox('THICKSTRT', brief='Cells with negative ICELLTYPE are confined'),
                CheckboxCheckbox('VARIABLECV', option2='DEWATERED'),
                Checkbox('PERCHED', brief='Use head minus bottom elevation of overlying, dewatered, convertible cell'),
                Checkbox3Fields(
                    'REWET',
                    option1='WETFCT',
                    type1='float',
                    value1=0.0,
                    option2='IWETIT',
                    type2='int',
                    value2=1,
                    option3='IHDWET',
                    type3='int',
                    value3=0
                ),
                CheckboxCheckbox(
                    'XT3D', brief='Use XT3D formulation', option2='RHS', brief2='Add XT3D terms to right-hand side'
                ),
                Checkbox(
                    'SAVE_SPECIFIC_DISCHARGE', brief='Save x, y, and z components of specific discharge to budget file'
                ),
                Checkbox('SAVE_SATURATION', brief='Save cell saturation to budget file'),
                Checkbox(
                    'K22OVERK',
                    brief='K22 (typically horizontal K in y direction) is a ratio and will be multiplied by K'
                ),
                Checkbox('K33OVERK', brief='K33 (typically vertical K) is a ratio and will be multiplied by K'),
                CheckboxField('TVK6 FILEIN', brief='Time-varying hydraulic conductivity (TVK) file', type_='str'),
                options_util.export_array_ascii_def(),
            ]
        )

    @override
    def get_tool_tip(self, tab_text: str) -> str:
        """Returns the tool tip that goes with the tab with the tab_name.

        Args:
            tab_text: Text of the tab

        Returns:
            (str): The tool tip.
        """
        txt = ''
        if tab_text == 'ICELLTYPE':
            txt = (
                'Flag for each cell that specifies how saturated thickness is treated. 0 means saturated thickness'
                ' is held constant; >0 means saturated thickness varies with computed head when head is below the'
                ' cell top; <0 means saturated thickness varies with computed head unless the THICKSTRT option is'
                ' in effect. When THICKSTRT is in effect, a negative value of icelltype indicates that saturated'
                ' thickness will be computed as STRT-BOT and held constant.'
            )
        elif tab_text == 'K':
            txt = (
                'Hydraulic conductivity. For the common case in which the user would like to specify the horizontal'
                ' hydraulic conductivity and the vertical hydraulic conductivity, then K should be assigned as the'
                ' horizontal hydraulic conductivity, K33 should be assigned as the vertical hydraulic conductivity,'
                ' and K22 and the three rotation angles should not be specified. When more sophisticated anisotropy'
                ' is required, then K corresponds to the K11 hydraulic conductivity axis. All included cells'
                ' (IDOMAIN > 0) must have a K value greater than zero.'
            )
        elif tab_text == 'K22':
            txt = (
                'Hydraulic conductivity of the second ellipsoid axis (or the ratio of K22/K if the K22OVERK option'
                ' is specified); for an unrotated case this is the hydraulic conductivity in the y direction. If'
                ' K22 is not included in the GRIDDATA block, then K22 is set equal to K. For a regular MODFLOW grid'
                ' (DIS Package is used) in which no rotation angles are specified, K22 is the hydraulic'
                ' conductivity along columns in the y direction. For an unstructured DISU grid, the user must'
                ' assign principal x and y axes and provide the angle for each cell face relative to the assigned x'
                ' direction. All included cells (IDOMAIN > 0) must have a K22 value greater than zero.'
            )
        elif tab_text == 'K33':
            txt = (
                'Hydraulic conductivity of the third ellipsoid axis (or the ratio of K33/K if the K33OVERK option'
                ' is specified); for an unrotated case, this is the vertical hydraulic conductivity. When'
                ' anisotropy is applied, K33 corresponds to the K33 tensor component. All included cells (IDOMAIN >'
                ' 0) must have a K33 value greater than zero.'
            )
        elif tab_text == 'ANGLE1':
            txt = (
                'A rotation angle of the hydraulic conductivity tensor in degrees. The angle represents the first'
                ' of three sequential rotations of the hydraulic conductivity ellipsoid. With the K11, K22, and K33'
                ' axes of the ellipsoid initially aligned with the x, y, and z coordinate axes, respectively,'
                ' ANGLE1 rotates the ellipsoid about its K33 axis (within the x - y plane). A positive value'
                ' represents counter-clockwise rotation when viewed from any point on the positive K33 axis,'
                ' looking toward the center of the ellipsoid. A value of zero indicates that the K11 axis lies'
                ' within the x - z plane. If ANGLE1 is not specified, default values of zero are assigned to'
                ' ANGLE1, ANGLE2, and ANGLE3, in which case the K11, K22, and K33 axes are aligned with the x, y,'
                ' and z axes, respectively.'
            )
        elif tab_text == 'ANGLE2':
            txt = (
                'A rotation angle of the hydraulic conductivity tensor in degrees. The angle represents the second'
                ' of three sequential rotations of the hydraulic conductivity ellipsoid. Following the rotation by'
                ' ANGLE1 described above, ANGLE2 rotates the ellipsoid about its K22 axis (out of the x - y plane).'
                ' An array can be specified for ANGLE2 only if ANGLE1 is also specified. A positive value of ANGLE2'
                ' represents clockwise rotation when viewed from any point on the positive K22 axis, looking toward'
                ' the center of the ellipsoid. A value of zero indicates that the K11 axis lies within the x - y'
                ' plane. If ANGLE2 is not specified, default values of zero are assigned to ANGLE2 and ANGLE3;'
                ' connections that are not user-designated as vertical are assumed to be strictly horizontal'
                ' Groundwater Flow (GWF) Model Input 47 (that is, to have no z component to their orientation); and'
                ' connection lengths are based on horizontal distances.'
            )
        elif tab_text == 'ANGLE3':
            txt = (
                'A rotation angle of the hydraulic conductivity tensor in degrees. The angle represents the third'
                ' of three sequential rotations of the hydraulic conductivity ellipsoid. Following the rotations by'
                ' ANGLE1 and ANGLE2 described above, ANGLE3 rotates the ellipsoid about its K11 axis. An array can'
                ' be specified for ANGLE3 only if ANGLE1 and ANGLE2 are also specified. An array must be specified'
                ' for ANGLE3 if ANGLE2 is specified. A positive value of ANGLE3 represents clockwise rotation when'
                ' viewed from any point on the positive K11 axis, looking toward the center of the ellipsoid. A'
                ' value of zero indicates that the K22 axis lies within the x - y plane.'
            )
        elif tab_text == 'WETDRY':
            txt = (
                'A combination of the wetting threshold and a flag to indicate which neighboring cells can cause a'
                ' cell to become wet. If WETDRY < 0, only a cell below a dry cell can cause the cell to become wet.'
                ' If WETDRY > 0, the cell below a dry cell and horizontally adjacent cells can cause a cell to'
                ' become wet. If WETDRY is 0, the cell cannot be wetted. The absolute value of WETDRY is the'
                ' wetting threshold. When the sum of BOT and the absolute value of WETDRY at a dry cell is equaled'
                ' or exceeded by the head at an adjacent cell, the cell is wetted. WETDRY must be specified if'
                ' “REWET” is specified in the OPTIONS block. If “REWET” is not specified in the options block, then'
                ' WETDRY can be entered, and memory will be allocated for it, even though it is not used.'
            )
        return txt
