"""SwiData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.griddata_base import GriddataBase
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import units_util
from xms.mf6.gui.options_defs import CheckboxButton, CheckboxField

# Constants used with SWI set up
SEA_LEVEL_CSV = 'SEA_LEVEL_CSV'
CREATE_GHB = 'CREATE_GHB'
GHB_CONDUCTANCE = 'GHB_CONDUCTANCE'
DEFAULT_GHB_CONDUCTANCE = 1.0e6
CREATE_DRN = 'CREATE_DRN'
DRN_CONDUCTANCE = 'DRN_CONDUCTANCE'
DEFAULT_DRN_CONDUCTANCE = 1.0e6
FIX_RCH = 'FIX_RCH'
CREATE_SALTWATER_HEAD_TVA = 'CREATE_SALTWATER_HEAD_TVA'


class SwiData(GriddataBase):
    """Data class to hold the info from a SWI package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'SWI6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'SWI6'
        self.add_block('GRIDDATA', ['ZETASTRT'])

    def get_units(self, array_name: str) -> str:
        """Returns the units string for the array.

        Args:
            array_name: The name of a array.

        Returns:
            (str): The units string like 'L' or 'L^3/T'.
        """
        match array_name:
            case 'ZETASTRT':
                units_spec = units_util.UNITS_LENGTH
            case _:  # pragma: no cover
                raise ValueError()
        return units_spec

    def is_int_array(self, array_name: str) -> bool:
        """Returns True if the array is integers.

        Args:
            array_name: The name of a array.

        Returns:
            (bool): True or False
        """
        match array_name:
            case 'ZETASTRT':
                return False
            case _:  # pragma: no cover
                raise ValueError()

    @override
    def is_required_array(self, array_name: str) -> bool:
        """Returns True if the array is required.

        Args:
            array_name: The name of a array.

        Returns:
            (bool): True or False
        """
        return array_name.upper() == 'ZETASTRT'

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Sea Water Intrusion (SWI) Package'

    # @overrides
    def _setup_options(self):
        """Returns the definition of all the available options.

        Returns:
            (OptionsBlock): See description.
        """
        return OptionsBlock(
            [
                CheckboxField('ZETA FILEOUT', brief='Save density to file', type_='str'),
                CheckboxButton(
                    'TVA6 FILEIN',
                    brief='Time-series files',
                    button_text='Files...',
                    button_method='on_btn_tva6_filein'
                ),
                CheckboxField('SALTWATER_HEAD', brief='Saltwater head value', type_='float', value=0.0),
            ]
        )

    @override
    def get_tool_tip(self, tab_text: str) -> str:
        """Returns the tool tip that goes with the tab with the tab_name.

        Args:
            tab_text: Text of the tab

        Returns:
            (str): The tool tip.
        """
        txt = ''
        if tab_text == 'ZETASTRT':
            txt = (
                'Initial (starting) zeta surface for each model cell—that is, interface elevation at the beginning of'
                ' the GWF Model simulation.'
            )
        return txt
