"""DspData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.griddata_base import GriddataBase
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import options_util, units_util
from xms.mf6.gui.options_defs import Checkbox


class DspData(GriddataBase):
    """Data class to hold the info from a DSP package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'DSP6'
        self.add_block('GRIDDATA', ['DIFFC', 'ALH', 'ALV', 'ATH1', 'ATH2', 'ATV'])

    def get_units(self, array_name: str) -> str:
        """Returns the units string for the array.

        Args:
            array_name (str): The name of a array.

        Returns:
            (str): The units string like 'L' or 'L^3/T'.
        """
        match array_name:
            case 'DIFFC':
                units_spec = units_util.UNITS_DIFFC
            case 'ALH' | 'ALV' | 'ATH1' | 'ATH2' | 'ATV':
                units_spec = units_util.UNITS_LENGTH
            case _:
                units_spec = ''  # This is an error
        return units_spec

    def is_int_array(self, array_name):
        """Returns True if the array is integers.

        Args:
            array_name (str): The name of a array.

        Returns:
            (bool): True or False
        """
        return False

    @override
    def is_required_array(self, array_name: str) -> bool:
        """Returns True if the array is required.

        Args:
            array_name (str): The name of a array.

        Returns:
            (bool): True or False
        """
        return False

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Dispersion (DSP) Package'

    # def map_info(self, feature_type):
    #     """Returns info needed for Map from Coverage.
    #
    #     Args:
    #         feature_type (str): 'points', 'arcs', or 'polygons'
    #
    #     Returns:
    #         (dict): Dict describing how to get the MODFLOW variable from the shapefile or att table fields.
    #     """
    #     if feature_type != 'polygons':
    #         return {}
    #     return {'HK': Mapping(mf_name='K'), 'VK': Mapping(mf_name='K33'), 'HANI': Mapping(mf_name='K22'),
    #             'VANI': Mapping(mf_name='K33')}

    @override
    def _setup_options(self) -> OptionsBlock:
        """Returns the definition of all the available options.

        Returns:
            See description.
        """
        return OptionsBlock(
            [
                Checkbox('XT3D_OFF', brief='Deactivate the xt3d method'),
                Checkbox('XT3D_RHS', 'Add xt3d terms to right-hand side, when possible'),
                options_util.export_array_ascii_def()
            ]
        )

    @override
    def get_tool_tip(self, tab_text: str) -> str:
        """Returns the tool tip that goes with the tab with the tab_name.

        Args:
            tab_text: Text of the tab

        Returns:
            (str): The tool tip.
        """
        txt = ''
        if tab_text == 'DIFFC':
            txt = 'Effective molecular diffusion coefficient.'
        elif tab_text == 'ALH':
            txt = (
                'Longitudinal dispersivity in horizontal direction. If flow is strictly horizontal, then this is the'
                ' longitudinal dispersivity that will be used. If flow is not strictly horizontal or strictly vertical,'
                ' then the longitudinal dispersivity is a function of both ALH and ALV. If mechanical dispersion is'
                ' represented (by specifying any dispersivity values) then this array is required.'
            )
        elif tab_text == 'ALV':
            txt = (
                'Longitudinal dispersivity in vertical direction. If flow is strictly vertical, then this is the'
                ' longitudinal dispersivity that will be used. If flow is not strictly horizontal or strictly vertical,'
                ' then the longitudinal dispersivity is a function of both ALH and ALV. If this value is not specified'
                ' and mechanical dispersion is represented, then this array is set equal to ALH.'
            )
        elif tab_text == 'ATH1':
            txt = (
                'Transverse dispersivity in horizontal direction. This is the transverse dispersivity value for the'
                ' second ellipsoid axis. If flow is strictly horizontal and directed in the x direction (along a row'
                ' for a regular grid), then this value controls spreading in the y direction. If mechanical dispersion'
                ' is represented (by specifying any dispersivity values) then this array is required.'
            )
        elif tab_text == 'ATH2':
            txt = (
                'Transverse dispersivity in horizontal direction. This is the transverse dispersivity value for the'
                ' third ellipsoid axis. If flow is strictly horizontal and directed in the x direction (along a row'
                ' for a regular grid), then this value controls spreading in the Z direction. If this value is not'
                ' specified and mechanical dispersion is represented, then this array is set equal to ATH1.'
            )
        elif tab_text == 'ATV':
            txt = (
                'Transverse dispersivity when flow is in vertical direction. If flow is strictly vertical and directed'
                ' in the z direction, then this value controls spreading in the x and y directions. If this value is'
                ' not specified and mechanical dispersion is represented, then this array is set equal to ATH2.'
            )
        return txt
