"""IcData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
from typing import Any

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules
from xms.guipy import file_io_util

# 4. Local modules
from xms.mf6.data.array_layer import ArrayLayer
from xms.mf6.data.griddata_base import GriddataBase
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.file_io import io_util
from xms.mf6.gui import options_util, units_util


class IcData(GriddataBase):
    """Data class to hold the info from a IC package file."""

    STRT_EQUALS_TOPS = 'STRT_EQUALS_TOPS'

    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'IC6'
        self.add_block('GRIDDATA', ['STRT'])

    def get_units(self, array_name: str) -> str:
        """Returns the units string for the array.

        Args:
            array_name (str): The name of an array.

        Returns:
            (str): The units string like 'L' or 'L^3/T'.
        """
        units_spec = ''
        match array_name:
            case 'STRT':
                if self.model:
                    if self.model.ftype == 'GWF6':
                        units_spec = units_util.UNITS_LENGTH
                    elif self.model.ftype == 'GWT6':
                        units_spec = units_util.UNITS_CONCENTRATION
                    elif self.model.ftype == 'GWE6':
                        units_spec = units_util.UNITS_TEMPERATURE
            case _:
                units_spec = ''  # This is an error
        return units_spec

    def is_int_array(self, array_name):
        """Returns True if the array is integers.

        Args:
            array_name (str): The name of an array.

        Returns:
            (bool): True or False
        """
        return array_name.upper() != 'STRT'

    @override
    def is_required_array(self, array_name: str) -> bool:
        """Returns True if the array is required.

        Args:
            array_name (str): The name of an array.

        Returns:
            (bool): True or False
        """
        return array_name.upper() == 'STRT'

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Initial Conditions (IC) Package'

    # @overrides
    def _setup_gms_options(self) -> dict[Any, Any]:
        """Some packages have additional GMS options, which this reads and returns or initializes to defaults.

        Returns:
            (OptionsBlock): See description.
        """
        if self._base.filename and self._base.model and self._base.model.ftype == 'GWF6':
            return self.read_or_init_gms_options(Path(self._base.filename).parent)
        return {}

    @override
    def _setup_options(self) -> OptionsBlock:
        """Some packages have additional GMS options, which this reads and returns or initializes to defaults.

        Returns:
            See description.
        """
        return OptionsBlock([options_util.export_array_ascii_def()])

    @classmethod
    def read_or_init_gms_options(cls, parent_dir: str | Path) -> dict[Any, Any]:
        """Returns the gms_options dict by reading it from disk or initializing it to defaults if the file isn't found.

        Args:
            parent_dir (str | Path): Path to the directory where the gms options file is located.

        Returns:
            See description.
        """
        gms_options_filepath = io_util.get_gms_options_filepath(parent_dir)
        gms_options = {}
        if gms_options_filepath.is_file():
            gms_options = file_io_util.read_json_file(gms_options_filepath)
        else:  # Initialize the options to our defaults
            # Set STRT_EQUALS_TOPS option to True if we're creating the file from scratch
            pattern = 'xms/mf6/components/resources/base_model/files'
            parent_path = Path(parent_dir)
            gms_options[cls.STRT_EQUALS_TOPS] = parent_path.match(pattern) or parent_path == Path('.')
        return gms_options

    def set_strt_equal_to_top(self, dis_data: GriddataBase) -> None:
        """Sets the STRT array to be the top elevations of the cells.

        Args:
            dis_data (GriddataBase): The DIS* package.
        """
        if self.model.ftype != 'GWF6':
            return

        # Get TOP, bottom, and STRT
        top_array = dis_data.block('GRIDDATA').array('TOP')
        bottom_name = 'BOT' if dis_data.ftype == 'DISU6' else 'BOTM'
        bottom_array = dis_data.block('GRIDDATA').array(bottom_name)
        strt_array = self.block('GRIDDATA').array('STRT')

        # Get STRT shape
        grid_info = self.grid_info()
        strt_count, strt_shape = ArrayLayer.number_of_values_and_shape(strt_array.layered, grid_info)

        if strt_array.layered:
            strt_array.layer(0).set_values(top_array.layer(0).get_values(), strt_shape, use_constant=True)
            if bottom_array.layered:
                for i in range(1, len(strt_array.layers)):
                    values = bottom_array.layer(i - 1).get_values()
                    strt_array.layer(i).set_values(values, strt_shape)
            else:
                values = bottom_array.layer(0).get_values()
                for i in range(1, len(strt_array.layers)):
                    start = strt_count * (i - 1)
                    stop = strt_count * i
                    strt_array.layer(i).set_values(values[start:stop], strt_shape)
        else:  # STRT is not layered
            # Create list of values from top and bottom
            values = top_array.get_values()
            values.extend(bottom_array.get_values()[:-len(values)])
            # Set STRT to values
            strt_array.set_values(values=values, shape=strt_shape, combine=False)

    @override
    def get_tool_tip(self, tab_text: str) -> str:
        """Returns the tool tip that goes with the tab with the tab_name.

        Args:
            tab_text: Text of the tab

        Returns:
            (str): The tool tip.
        """
        txt = ''
        if tab_text == 'STRT':
            txt = (
                'Initial (starting) head—that is, head at the beginning of the GWF Model simulation. STRT must be'
                ' specified for all simulations, including steady-state simulations. One value is read for every model'
                ' cell. For simulations in which the first stress period is steady state, the values used for STRT'
                ' generally do not affect the simulation (exceptions may occur if cells go dry and (or) rewet). The'
                ' execution time, however, will be less if STRT includes hydraulic heads that are close to the'
                ' steady-state solution. A head value lower than the cell bottom can be provided if a cell should start'
                ' as dry.'
            )
        return txt
