"""MfsimData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path
from typing import Any

# 2. Third party modules

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as fs
from xms.guipy import file_io_util

# 4. Local modules
from xms.mf6.components.xms_data import XmsData
from xms.mf6.data import data_util
from xms.mf6.data.base_file_data import BaseFileData
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.file_io import io_util
from xms.mf6.gui.options_defs import Checkbox, CheckboxComboBox, CheckboxField


class SolutionGroup:
    """A solution group."""
    def __init__(self):
        """Initializes the class."""
        self.use_mxiter = False
        self.mxiter = 1
        self.ims_list = []  # ImsData


class MfsimData(BaseFileData):
    """A MODFLOW 6 simulation."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'MFSIM6'
        self.mfsim = self
        self.tdis = None  # TdisData
        self.models = []  # List of GwfModels
        self.exchanges = []  # List of exchanges
        self.solution_groups = []  # List of SolutionGroup
        self.gui_edit_active = False  # being edited by gui
        self.xms_data = XmsData(None)  # Has query
        self.can_import = False
        self.can_export = False  # Menu command is still available. Is skipped dialog export for now

    def add_model(self, model):
        """Adds the GWF model to the list.

        Args:
            model: The GwfModel.
        """
        self.models.append(model)

    def dialog_title(self):
        """Returns the title to show in the dialog.

        You should override this method.

        Returns:
            (str): The dialog title.
        """
        return 'Simulation Options'

    @classmethod
    def read_or_init_gms_options(cls, mfsim_dir: str | Path) -> dict[Any, Any]:
        """Returns the gms_options dict by reading it from disk or initializing it to defaults if the file isn't found.

        Args:
            mfsim_dir (str | Path): Path to the directory where the mfsim.nam file is located.

        Returns:
            See description.
        """
        gms_options_filepath = io_util.get_gms_options_filepath(mfsim_dir)
        gms_options = {}
        if gms_options_filepath.is_file():
            gms_options = file_io_util.read_json_file(gms_options_filepath)
        else:  # Initialize the options to our defaults
            gms_options['USE_OPEN_CLOSE'] = True
            gms_options['USE_INPUT_DIR'] = True
            gms_options['USE_OUTPUT_DIR'] = True
            gms_options['AUTO_FILE_NAMING'] = True
        return gms_options

    def model_from_mname(self, mname):
        """Returns the model with the given mname by searching the mfsim, or None if not found.

        Args:
            mname (str): Model mname.

        Returns:
            The package data object.
        """
        for model in self.models:
            if model.mname == mname:
                return model
        return None

    def model_from_filename(self, filename):
        """Returns the model with the given filename by searching the mfsim, or None if not found.

        Args:
            filename (str): Path of the model name file.

        Returns:
            The package data object.
        """
        for model in self.models:
            if fs.paths_are_equal(os.path.basename(model.filename), filename):
                return model
        return None

    def model_from_component_uuid(self, component_uuid):
        """Returns the model with the given filename by searching the mfsim, or None if not found.

        Args:
            component_uuid (str): uuid of component.

        Returns:
            The package data object.
        """
        for model in self.models:
            model_uuid = io_util.uuid_from_path(model.filename)
            if model_uuid == component_uuid:
                return model
        return None

    def _paths_are_equal(self, path1, path2, testing):
        if testing:
            return fs.paths_are_equal(os.path.basename(path1), path2)
        else:
            return fs.paths_are_equal(path1, path2)

    def package_from_filename(self, filename, testing=False):
        """Returns the package with the given filename by searching the mfsim, or None if not found.

        Args:
            filename (str): Path of the package file.
            testing (bool): True if we're testing.

        Returns:
            The package data object.
        """
        if self._paths_are_equal(self.filename, filename, testing):
            return self
        if self.tdis and self._paths_are_equal(self.tdis.filename, filename, testing):
            return self.tdis
        for model in self.models:
            if self._paths_are_equal(model.filename, filename, testing):
                return model
            package = model.package_from_filename(filename, testing)
            if package:
                return package
        for exchange in self.exchanges:
            if self._paths_are_equal(exchange.filename, filename, testing):
                return exchange
        for group in self.solution_groups:
            for ims in group.ims_list:
                if self._paths_are_equal(ims.filename, filename, testing):
                    return ims
        return None

    def packages_from_ftype(self, ftype: str) -> list[BaseFileData]:
        """Returns the packages with the given ftype.

        Args:
            ftype: The file type used in the GWF name file (e.g. 'WEL6')

        Returns:
            The list of packages.
        """
        packages = []
        if ftype == 'MFSIM6':
            packages.append(self)
        elif self.tdis and ftype == 'TDIS6':
            packages.append(self.tdis)
        elif ftype in data_util.exchange_ftypes():
            for exchange in self.exchanges:
                if exchange.exgtype.upper() == ftype:
                    packages.append(exchange)
        elif ftype == 'IMS6':
            for group in self.solution_groups:
                packages.extend(group.ims_list)
        else:
            for model in self.models:
                if ftype == model.ftype:
                    packages.append(model)
                else:
                    packages.extend(model.packages_from_ftype(ftype))
        return packages

    def dir_(self):
        """Returns the path to the directory where the mfsim.nam is.

        Returns:
            (str): See description.
        """
        return os.path.dirname(self.filename)

    # @overrides
    def _setup_options(self):
        """Returns the definition of all the available options.

        Returns:
            (OptionsBlock): See description.
        """
        return OptionsBlock(
            [
                Checkbox('CONTINUE', brief='Continue even if one or more solutions do not converge'),
                Checkbox('NOCHECK', brief='Model input check routines should not be called prior to each time step'),
                CheckboxComboBox(
                    'MEMORY_PRINT_OPTION',
                    brief='Print detailed memory manager usage',
                    items=['NONE', 'SUMMARY', 'ALL'],
                    value='NONE'
                ),
                CheckboxComboBox(
                    'PROFILE_OPTION',
                    brief='Performance profiling and reporting',
                    items=['NONE', 'SUMMARY', 'DETAIL'],
                    value='NONE'
                ),
                CheckboxField(
                    'MAXERRORS',
                    brief='Maximum number of errors that will be stored and printed',
                    type_='int',
                    value=50
                ),
                Checkbox('PRINT_INPUT', brief='Print simulation input summaries to the list file (mfsim.lst)'),
            ]
        )
