"""LakWriter class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import shlex
import sys

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.file_io import io_util
from xms.mf6.file_io.list_package_writer import ListPackageWriter


class LakWriter(ListPackageWriter):
    """Writes a LAK package file."""
    def __init__(self):
        """Initializes the class."""
        super().__init__()
        self.noutlets = 0
        self.ntables = 0

    def _write_table_files(self):
        """Copy the lak table files to the appropriate location."""
        fname = self._data.list_blocks['TABLES']
        with open(fname, 'r') as f:
            lines = f.readlines()
        with open(fname, 'w') as f:
            for line in lines:
                words = shlex.split(line, posix="win" not in sys.platform)
                words[3] = words[3].strip('\"')
                self.write_file_list(
                    [words[3]], words[1], 'FILEIN', f, self._data.filename, self._writer_options, int(words[0])
                )

    def _write_dimensions(self, fp):
        """Writes the dimensions block.

        Args:
            fp (_io.TextIOWrapper): The file.
        """
        self._data.maxbound = 0
        if 'PACKAGEDATA' in self._data.list_blocks:
            self._data.maxbound = io_util.count_max_line([self._data.list_blocks['PACKAGEDATA']])

        if 'OUTLETS' in self._data.list_blocks:
            self.noutlets = io_util.count_max_line([self._data.list_blocks['OUTLETS']])

        if 'TABLES' in self._data.list_blocks:
            self.ntables = io_util.count_max_line([self._data.list_blocks['TABLES']])

        fp.write('\n')
        fp.write('BEGIN DIMENSIONS\n')
        fp.write(f'{io_util.mftab}NLAKES {self._data.maxbound}\n')
        if self.noutlets > 0:
            fp.write(f'{io_util.mftab}NOUTLETS {self.noutlets}\n')
        if self.ntables > 0:
            fp.write(f'{io_util.mftab}NTABLES {self.ntables}\n')
        fp.write('END DIMENSIONS\n')

    def _write_package(self, data):
        """Writes the package file.

        Saves stress period data in external files using the OPEN/CLOSE option.
        """
        self._data = data
        with open(self._data.filename, 'w') as fp:
            self._write_comments(fp)
            self._write_options(fp)
            self._write_dimensions(fp)
            self._write_list_block('PACKAGEDATA', fp)
            self._write_list_block('CONNECTIONDATA', fp)
            if self.ntables > 0:
                self._write_table_files()
                self._write_list_block('TABLES', fp)
            if self.noutlets > 0:
                self._write_list_block('OUTLETS', fp)
            self._write_stress_periods(fp)
