"""Reads the PEST obs results."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass, field
from datetime import datetime

# 2. Third party modules
from orjson import orjson

# 3. Aquaveo modules

# 4. Local modules

# Type aliases

# ---- Aliases for Obs

ObsVals = dict[str:float, str:float, str:bool, str:float]
"""ObsVals: all the values for an object, e.g. {'computed': 1.0, 'flow': False, 'observed': 1.0, 'weight': 1.2}."""

Observation = dict[datetime:ObsVals]
"""Observation: datetime and ObsVals, e.g. {'1950-01-01 00:00:00': ObsVals, '1950-01-02 00:00:00': ObsVals ...}."""

Obs = dict[str:Observation]
"""Obs: the object aliases and Observation, e.g. {'POINT_#1': Observation, 'POINT_#2': Observation ...}."""

# ---- Aliases for B2Map

ObjGeom = dict[str:list[float], str:float]
"""ObjGeom: object geometry and interval, e.g. {'geometry': [10.0, 10.0, 0.0], 'interval': 1.5}."""

AliasGeom = dict[str:ObjGeom]
"""AliasGeom: object alias and ObjGeom, e.g. {'POINT_#1': ObjGeom, 'POINT_#2': ObjGeom, ...}."""

Features = dict[str:AliasGeom, str:AliasGeom, str:AliasGeom, str:AliasGeom]
"""Features: feature type: AliasGeom, e.g. {'points': AliasGeom, 'arcs': {}, ...}."""

ObsCovData = dict[str:str, str:Features]
"""ObsCovData: cov tree path and Features, e.g. {'coverage_tree_path': 'Project/Map', 'features': Features}."""

B2Map = dict[str:ObsCovData]
"""B2Map: coverage uuid and ObsCovData, e.g. {'84c9cae0-2cd0-4cc4-b981-a604dd520d17': ObsCovData ...}"""


@dataclass
class ObsResults:
    """ObsResults: Obs and B2Map, e.g. {'obs': Obs, 'b2map': B2Map, 'dset_uuid': uuid}."""
    obs: Obs = field(default_factory=dict)
    b2map: B2Map = field(default_factory=dict)
    dset_uuid: str = ''

    def to_json(self) -> str:
        """Return the data as a json string.

        Returns:
            See description.
        """
        # Use orjson because it can handle dataclasses and datetime keys
        sbytes = orjson.dumps(self, option=orjson.OPT_NON_STR_KEYS | orjson.OPT_INDENT_2)
        return sbytes.decode('utf-8')
