"""TdisWriter class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.file_io import io_util
from xms.mf6.file_io.package_writer_base import PackageWriterBase


class TdisWriter(PackageWriterBase):
    """Writes a tdis package file."""
    def __init__(self):
        """Initializes the class."""
        super().__init__()

    def _write_option(self, key, fp, options_dict):
        """Writes the options block.

        Args:
            key: The key.
            fp (file): The file.
            options_dict (dict): Optional dict containing the options.
        """
        if key == 'START_DATE_TIME':
            start_date_time = self._data.options_block.get('START_DATE_TIME', '')
            if start_date_time:
                start_date_time = start_date_time.strip('\'"')
                word = f'\'{start_date_time}\'' if ' ' in start_date_time else f'{start_date_time}'
                fp.write(f'{io_util.mftab}START_DATE_TIME {word}\n')
        else:
            super()._write_option(key, fp, options_dict)

    def _write_dimensions(self, fp):
        """Writes the dimensions block.

        Args:
            fp (file): The file.
        """
        fp.write('\n')
        fp.write('BEGIN DIMENSIONS\n')
        fp.write(io_util.mftab)
        fp.write('NPER {}\n'.format(self._data._nper))
        fp.write('END DIMENSIONS\n')

    def _write_period_data(self, fp):
        """Writes all the stress period data.

        Args:
            fp (file): The file.
        """
        fp.write('\n')
        fp.write('BEGIN PERIODDATA\n')
        for sp in range(1, self._data._nper + 1):
            fp.write(io_util.mftab)
            fp.write(
                '{} {} {}\n'.format(
                    self._data.period_df.iat[sp - 1, 0], self._data.period_df.iat[sp - 1, 1],
                    self._data.period_df.iat[sp - 1, 2]
                )
            )
        fp.write('END PERIODDATA\n')

    def _write_package(self, data):
        """Writes the package file."""
        self._data = data
        with open(self._data.filename, 'w') as fp:
            self._write_comments(fp)
            self._write_options(fp)
            self._write_dimensions(fp)
            self._write_period_data(fp)
