"""DataTabsWidget class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import (QHBoxLayout, QTabWidget, QToolBar, QVBoxLayout, QWidget)

# 3. Aquaveo modules
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate
from xms.guipy.widgets import widget_builder
from xms.guipy.widgets.widget_builder import ActionRec

# 4. Local modules
from xms.mf6.file_io import io_util
from xms.mf6.gui.resources.svgs import FILTER_SVG
from xms.mf6.gui.widgets.list_block_table_widget import ListBlockTableWidget
from xms.mf6.gui.widgets.period_list_widget import PeriodListWidget


class DataTabsWidget(QWidget):
    """A widget with a tab control that holds list blocks and a PERIODS block.

    List blocks are probably represented using ListBlockTableWidget.
    """
    def __init__(self, dlg_input, parent=None):
        """Initializes the class, sets up the ui, and loads the simulation.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window. Assumed to be a dialog.
        """
        super().__init__(parent)
        self.dlg_input = dlg_input  # ArrayPackageData. Gets created in load_package
        self.loading = False  # True in load_arrays
        self.ui_tabs = {}  # Widgets added to the tab control
        self.ui_sp_list_widget = None  # PeriodListWidget
        self.uix = {}  # dictionary of widgets for sections
        self.actions = {}

    def do_enabling(self):
        """Enables and disables the widgets appropriately."""
        tab_index = self.uix['tab_widget'].currentIndex()
        if tab_index < 0:
            return

        # Enable/disable toolbar actions
        tab_name = self.current_tab_name()
        if tab_name in self.ui_tabs:
            self.ui_tabs[tab_name]['tbl'].btn_add_rows.setEnabled(not self.dlg_input.locked)

    def setup(self):
        """Sets up the DATA section."""
        vlayout = QVBoxLayout()
        self.setLayout(vlayout)

        w = self.uix['tab_widget'] = QTabWidget()
        vlayout.addWidget(w)
        self.create_tabs()

        self.setup_signals()

    def setup_signals(self):
        """Sets up any needed signals."""
        self.uix['tab_widget'].currentChanged.connect(self.on_tab_changed)

    def create_list_block_tabs(self):
        """Creates the tabs for the list blocks."""
        for name in self.dlg_input.data.list_blocks:
            widgets = {}
            page = QWidget()
            vlayout = QVBoxLayout()

            if name == self.dlg_input.data.block_with_cellids:
                self._create_toolbar(name, page, vlayout, widgets)

            # Table
            widgets['tbl'] = ListBlockTableWidget(parent=self, dlg_input=self.dlg_input, label='')
            vlayout.addWidget(widgets['tbl'])

            # Add combo boxes
            delegate_info = self.dlg_input.data.get_column_delegate_info(name)
            if delegate_info:
                delegate = QxCbxDelegate(self)
                for column in delegate_info:
                    delegate.set_strings(column[1])
                    widgets['tbl'].table.setItemDelegateForColumn(column[0], delegate)

            page.setLayout(vlayout)
            self.uix['tab_widget'].addTab(page, name)
            self.ui_tabs[name] = widgets

            self.get_table_widget(name).load_list_block(name)

    def _create_toolbar(self, name: str, page: QWidget, vlayout: QVBoxLayout, widgets: dict) -> None:
        """Create a toolbar above the table.

        Args:
            name: List block name
            page: The QWidget for the tab.
            vlayout: Vertical layout on the page.
            widgets: Dict of widgets for the page.
        """
        hlayout = QHBoxLayout()
        vlayout.addLayout(hlayout)
        tool_bar = widgets['toolbar'] = QToolBar(page)
        hlayout.addWidget(tool_bar)
        filter_ = ActionRec((FILTER_SVG, FILTER_SVG), 'Filter on Selected Cells', self.on_btn_filter, True)
        button_list = [filter_]
        self.actions[name] = widget_builder.setup_toolbar(tool_bar, button_list)
        checked = self.dlg_input.filter_on_selected_cells and len(self.dlg_input.selected_cells)
        self.actions[name][FILTER_SVG].setChecked(checked)

    def create_period_tab(self):
        """Creates the tab for the stress periods."""
        self.ui_sp_list_widget = PeriodListWidget(self.dlg_input.data, self.dlg_input, self)
        self.uix['tab_widget'].addTab(self.ui_sp_list_widget, 'PERIODS')

    def create_tabs(self):
        """Creates the tabs for the dialog."""
        self.loading = True
        self.create_list_block_tabs()
        self.create_period_tab()
        self.loading = False
        self.uix['tab_widget'].setCurrentIndex(0)

    def current_tab_name(self):
        """Returns the name of the current tab."""
        tab_index = self.uix['tab_widget'].currentIndex()
        tab_name = self.uix['tab_widget'].tabText(tab_index)
        return tab_name

    def on_tab_changed(self):
        """Called when the current tab changes."""
        if not self.loading:  # This is here to stop it while we're building the tabs
            self.do_enabling()

    def change_list_block_columns(self, block):
        """Changes the columns in the list block.

        Args:
            block: The block.
        """
        auxiliary_list = self.dlg_input.data.options_block.get('AUXILIARY', [])
        use_aux = bool(auxiliary_list)
        table_widget = self.get_table_widget(block)
        if table_widget:
            table_widget.change_columns(block, use_aux)

    def on_btn_filter(self):
        """Called when the filter button is clicked."""
        self.dlg_input.filter_on_selected_cells = not self.dlg_input.filter_on_selected_cells
        table_widget = self.get_table_widget(self.current_tab_name())
        if table_widget:
            table_widget.load_list_block(self.current_tab_name())

    def get_table_widget(self, block):
        """Returns the ListBlockTableWidget from the given block.

        Args:
            block: The block.

        Returns:
            (ListBlockTableWidget): The table widget.
        """
        if block in self.ui_tabs and 'tbl' in self.ui_tabs[block]:
            return self.ui_tabs[block]['tbl']
        return None

    def clean_up_temp_files(self):
        """Deletes all temp files."""
        io_util.clean_up_temp_files(self.dlg_input.data.period_files.values())

    def close_db(self):
        """Closes the database connections."""
        self.ui_sp_list_widget.close_db()

    def accept(self):
        """Called when user hits the OK button. Saves the widget state and closes the dialog."""
        if not self.dlg_input.locked:
            self.ui_sp_list_widget.accept()
        self.close_db()  # Do this whether dialog is locked or not

    def reject(self):
        """Called when the user clicks Cancel."""
        self.ui_sp_list_widget.reject()
        self.clean_up_temp_files()
