"""DefineStressPeriodDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QDialog

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.mf6.gui.define_stress_period_dialog_ui import Ui_DefineStressPeriodDialog


class DefineStressPeriodDialog(XmsDlg):
    """Dialog asking how to define a stress period.

    Copy from previous, copy from next, or leave empty.
    """
    def __init__(self, prev_sp, next_sp, parent=None):
        """Initializes the dialog.

        Args:
            prev_sp (int): The closest previously defined stress period.
            next_sp (int): The closest next defined stress period.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(parent, 'xms.mf6.gui.define_stress_period_dialog')

        self.ui = Ui_DefineStressPeriodDialog()
        self.ui.setupUi(self)

        if prev_sp == -1:
            self.ui.rbt_copy_prev.setText('Copy from previous period')
            self.ui.rbt_copy_prev.setEnabled(False)
        else:
            self.ui.rbt_copy_prev.setText(f'Copy from previous period ({prev_sp})')
            self.ui.rbt_copy_prev.setChecked(True)

        if next_sp == -1:
            self.ui.rbt_copy_next.setText('Copy from next period')
            self.ui.rbt_copy_next.setEnabled(False)
        else:
            self.ui.rbt_copy_next.setText(f'Copy from next period ({next_sp})')
            self.ui.rbt_copy_next.setChecked(prev_sp == -1)

        self.ui.buttonBox.accepted.connect(self.accept)
        self.ui.buttonBox.rejected.connect(self.reject)

    def get_choice(self):
        """Returns the choice the user made.

        Returns:
            (int): 0 = previous, 1 = next, 2 = empty
        """
        if self.ui.rbt_copy_prev.isChecked():
            return 0
        elif self.ui.rbt_copy_next.isChecked():
            return 1
        else:
            return 2


def run(prev_sp, next_sp, parent=None):
    """Runs the dialog.

    Args:
        prev_sp (int): The closest previously defined stress period.
        next_sp (int): The closest next defined stress period.
        parent (Something derived from QWidget): The parent window.

    Returns:
         (int): -1 = Cancel, 0 = previous, 1 = next, 2 = empty
    """
    dialog = DefineStressPeriodDialog(prev_sp, next_sp, parent)
    if dialog.exec() == QDialog.Accepted:
        return dialog.get_choice()
    else:
        return -1
