"""DisvDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QLabel

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.gui.dis_dialog import DisDialog
from xms.mf6.gui.dis_dialog_base import create_plain_text_edit, fill_edt_with_file


class DisvDialog(DisDialog):
    """A dialog used for the DISV package."""
    def __init__(self, dlg_input, parent=None):
        """Initializes the class, sets up the ui, and loads the package.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, parent)

    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections and self.default_sections should be set here.
        """
        self.sections = ['COMMENTS', 'OPTIONS', 'DIMENSIONS', 'GRIDDATA', 'VERTICES', 'CELL2D']
        self.default_sections = ['GRIDDATA']

    def setup_dimensions(self):
        """Sets up the dimensions section."""
        name = 'DIMENSIONS'
        self.add_group_box_to_scroll_area(name)

        # NLAY
        self.uix[name]['txt_nlay'] = QLabel('Number of layers (NLAY):')
        self.uix[name]['layout'].addWidget(self.uix[name]['txt_nlay'])

        # NCPL
        self.uix[name]['txt_ncpl'] = QLabel('Number of cells per layer (NCPL):')
        self.uix[name]['layout'].addWidget(self.uix[name]['txt_ncpl'])

        # NVERT
        self.uix[name]['txt_nvert'] = QLabel('Number of vertices (NVERT):')
        self.uix[name]['layout'].addWidget(self.uix[name]['txt_nvert'])

        self.uix[name]['txt_nlay'].setText(f'Number of layers (NLAY): {self.dlg_input.data.grid_info().nlay}')
        self.uix[name]['txt_ncpl'].setText(f'Number of cells per layer (NCPL): {self.dlg_input.data.grid_info().ncpl}')
        self.uix[name]['txt_nvert'].setText(f'Number of vertices (NVERT): {self.dlg_input.data.grid_info().nvert}')

    def setup_vertices(self):
        """Sets up the VERTICES."""
        name = 'VERTICES'
        self.add_group_box_to_scroll_area(name)

        # read only message
        txt = 'VERTICES is defined by the grid and is not editable.'
        self.uix[name]['txt_read_only'] = QLabel(txt)
        self.uix[name]['txt_read_only'].setStyleSheet('font-weight: bold; color: black')
        self.uix[name]['layout'].addWidget(self.uix[name]['txt_read_only'])

        # Add widgets
        edt_vertices = self.uix[name]['edt_vertices'] = create_plain_text_edit()
        self.uix[name]['layout'].addWidget(edt_vertices)

        if 'VERTICES' in self.dlg_input.data.list_blocks:
            filepath = self.dlg_input.data.list_blocks['VERTICES']
            fill_edt_with_file(filepath, edt_vertices, '')

    def setup_cell2d(self):
        """Sets up the CELL2D stuff."""
        name = 'CELL2D'
        self.add_group_box_to_scroll_area(name)

        # read only message
        txt = 'CELL2D is defined by the grid and is not editable.'
        self.uix[name]['txt_read_only'] = QLabel(txt)
        self.uix[name]['txt_read_only'].setStyleSheet('font-weight: bold; color: black')
        self.uix[name]['layout'].addWidget(self.uix[name]['txt_read_only'])

        # Add widgets
        edt_cell2d = self.uix[name]['edt_cell2d'] = create_plain_text_edit()
        self.uix[name]['layout'].addWidget(self.uix[name]['edt_cell2d'])

        if 'CELL2D' in self.dlg_input.data.list_blocks:
            filepath = self.dlg_input.data.list_blocks['CELL2D']
            fill_edt_with_file(filepath, edt_cell2d, '')

    def setup_section(self, section_name):
        """Sets up a block of widgets.

        Args:
            section_name (str): name of the block
        """
        if section_name == 'VERTICES':
            self.setup_vertices()
        elif section_name == 'CELL2D':
            self.setup_cell2d()
        else:
            super().setup_section(section_name)
