"""ExportSimulationDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QCheckBox, QDialog, QFileDialog, QGroupBox, QVBoxLayout, QWidget

# 3. Aquaveo modules
from xms.guipy.dialogs import xms_parent_dlg
from xms.guipy.widgets import widget_builder

# 4. Local modules


class ExportSimulationDialog(QFileDialog):
    """A custom file dialog."""
    def __init__(self, parent, directory, options):
        """Initializes the class.

        Args:
            parent: Parent window.
            directory (str): Starting directory.
            options: Dict containing the export options.
        """
        super().__init__(parent=parent, directory=directory, caption='Export Simulation')

        self.options = options
        self.uix = {}

        self.setAcceptMode(QFileDialog.AcceptSave)
        self.selectFile('mfsim.nam')
        self.setOption(QFileDialog.DontResolveSymlinks)
        self.setOption(QFileDialog.DontUseNativeDialog)
        self.setFileMode(QFileDialog.AnyFile)
        self._add_export_options()
        self._make_file_name_read_only()
        xms_parent_dlg.add_process_id_to_window_title(self)

    def _add_export_options(self):
        """Adds the Export Options button to the dialog."""
        layout = self.layout()
        widget_builder.insert_gridlayout_rows(0, 1, layout)  # Insert 3 rows to make room for our options

        w = self.uix['grp_options'] = QGroupBox('Export options')
        self.uix['layout'] = QVBoxLayout()
        self.uix['grp_options'].setLayout(self.uix['layout'])
        layout.addWidget(w, 0, 0, 1, layout.columnCount())

        w = self.uix['chk_open_close'] = QCheckBox('Use external files (OPEN/CLOSE) for list and array input')
        self.uix['layout'].addWidget(w)
        w.clicked.connect(self._on_chk_open_close)

        w = self.uix['chk_use_input_dir'] = QCheckBox('Use "input" subdirectory for external files')
        self.uix['layout'].addWidget(w)

        w = self.uix['chk_use_output_dir'] = QCheckBox('Use "output" subdirectory for solution files')
        self.uix['layout'].addWidget(w)

        self.uix['chk_open_close'].setChecked(self.options.get('USE_OPEN_CLOSE', False))
        self.uix['chk_use_input_dir'].setChecked(self.options.get('USE_INPUT_DIR', False))
        self.uix['chk_use_output_dir'].setChecked(self.options.get('USE_OUTPUT_DIR', False))

    def _make_file_name_read_only(self):
        """Makes the file name edit field read only, because it must be 'mfsim.nam'."""
        file_name_edit = self.findChild(QWidget, 'fileNameEdit')
        if file_name_edit:
            widget_builder.make_lineedit_readonly(file_name_edit)

    def update_options(self, gms_options):
        """Updates the options from the dialog controls.

        Args:
            gms_options: Stores the options.
        """
        gms_options['USE_OPEN_CLOSE'] = self.uix['chk_open_close'].isChecked()
        gms_options['USE_INPUT_DIR'] = self.uix['chk_use_input_dir'].isChecked()
        gms_options['USE_OUTPUT_DIR'] = self.uix['chk_use_output_dir'].isChecked()

    def _on_chk_open_close(self):
        self.uix['chk_use_input_dir'].setEnabled(self.uix['chk_open_close'].isChecked())

    def exec(self):
        """Override the base class method."""
        from xms.api.dmi import XmsEnvironment as XmEnv
        if XmEnv.xms_environ_running_tests() == 'TRUE':
            self.accept()
            return QDialog.Accepted
        else:
            return super().exec_()
