"""FlowBudgetDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from enum import IntEnum
import locale
import sys

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QHeaderView, QTableWidgetItem, QWidget

# 3. Aquaveo modules
from xms.api.tree import TreeNode
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.widgets import widget_builder

# 4. Local modules
from xms.mf6.gui import gui_util
from xms.mf6.gui.flow_budget_dialog_ui import Ui_flow_budget_dialog


class BudgetColumn(IntEnum):
    """The table columns."""
    BC_LABELS = 0
    BC_IN = 1
    BC_OUT = 2
    BC_END = 3


class FlowBudgetDialog(XmsDlg):
    """Displays the flow budget.

    All math should be done in FlowBudgetCalculator.
    """

    SOURCES_SINKS = 'Sources/Sinks'
    SELECTED_ZONE = 'Selected Zone'

    def __init__(self, flow_budgets: dict, initial_model_node: TreeNode, prec: int, parent: QWidget) -> None:
        """Initializes the class.

        Args:
            flow_budgets (dict): Dict of model node -> flow budget.
            initial_model_node: Tree node of initial model.
            prec (int): Precision, i.e. number of digits to the right of the decimal to show.
            parent (QWidget): The parent widget
        """
        super().__init__(parent, 'xms.mf6.gui.flow_budget_dialog')
        self.ui = Ui_flow_budget_dialog()
        self.ui.setupUi(self)

        locale.setlocale(locale.LC_ALL, '')  # Use the local locale settings

        # Member variables
        self.help_getter = gui_util.help_getter('xms.mf6.gui.flow_budget_dialog')
        self._flow_budgets = flow_budgets
        self._flow_budget = None

        self._setup_model_cbx(initial_model_node)
        self._set_precision_spin_box(prec)
        self._on_model_changed()

        # Signals
        self.ui.btn_box.helpRequested.connect(self.help_requested)
        self.ui.spn_precision.valueChanged.connect(self._on_spn_precision)

    def _setup_table(self) -> None:
        """Sets up the table."""
        self.ui.tbl.setRowCount(0)
        self.ui.tbl.setColumnCount(BudgetColumn.BC_END)
        self._do_headers()
        self._add_sources_sinks()
        self._add_blank_row()
        self._add_selected_zone()
        self._add_blank_row()
        self._add_total_flow()
        self._add_blank_row()
        self._add_summary()
        self._size_rows_and_columns()

    def _setup_model_cbx(self, init_model_node: TreeNode) -> None:
        """Setup the model combo box.

        Args:
            init_model_node: The initial model node.
        """
        # Populate the model combo box
        current_index = -1
        for model_node in self._flow_budgets.keys():
            self.ui.cbx_model.addItem(model_node.name, model_node)
            if model_node == init_model_node:
                current_index = self.ui.cbx_model.count() - 1
        self.ui.cbx_model.setCurrentIndex(current_index)

        # Connect signal and disable if only one model
        self.ui.cbx_model.currentIndexChanged.connect(self._on_model_changed)
        if len(self._flow_budgets) <= 1:
            self.ui.cbx_model.setEnabled(False)

    def _on_model_changed(self) -> None:
        """Called when the model combo box changes."""
        model_node = self.ui.cbx_model.currentData()
        self._flow_budget = self._flow_budgets[model_node]

        # Enable stuff
        data_exists = self._flow_budget is not None
        self.ui.vlay_table_wid.setVisible(data_exists)
        self.ui.vlay_no_data_wid.setVisible(not data_exists)
        if not data_exists:
            self.ui.lbl_no_data.setText(f'No flow budget data for model "{model_node.name}".')

        if self._flow_budget:
            self._setup_table()
            self._make_read_only()
            self._set_selected_cells_label()
            self._set_time_label()

    def _set_selected_cells_label(self) -> None:
        """Sets the label that says how many selected cells there are."""
        selected_cells_count = self._flow_budget['meta']['selected_cells_count']
        text = '0 (data for all cells is displayed)' if selected_cells_count == 0 else str(selected_cells_count)
        self.ui.lbl_selected_cells.setText(f'Number of selected cells: {text}')

    def _set_time_label(self) -> None:
        """Sets the label that says what time we are displaying data for."""
        time = self._flow_budget['meta']['time']
        self.ui.lbl_time_step.setText(f'Time: {time}')

    def _set_precision_spin_box(self, prec: int) -> None:
        """Sets up the precision spin box."""
        self.ui.spn_precision.setValue(prec)

    def _on_spn_precision(self):
        """Called when the number of precision changes."""
        self._setup_table()

    def _from_float(self, value: float) -> str:
        """Returns a string of the float with the correct precision."""
        prec = self.ui.spn_precision.value()
        format_string = f'%.{prec}f'
        # return locale.format(format_string, value, grouping=True)
        return locale.format_string(format_string, value, grouping=True)

    def _do_headers(self) -> None:
        """Adds the table header."""
        self.ui.tbl.verticalHeader().hide()
        self.ui.tbl.setHorizontalHeaderItem(BudgetColumn.BC_LABELS, QTableWidgetItem(''))
        widget_builder.style_table_view(self.ui.tbl)

        item = QTableWidgetItem('Flow In')
        font = item.font()
        font.setBold(True)
        item.setFont(font)
        self.ui.tbl.setHorizontalHeaderItem(BudgetColumn.BC_IN, item)

        item = QTableWidgetItem('Flow Out')
        item.setFont(font)
        self.ui.tbl.setHorizontalHeaderItem(BudgetColumn.BC_OUT, item)

    def _add_budgets(self, title: str) -> None:
        """Adds rows.

        Args:
            title (str): Title of the section.
        """
        row = self._add_row()
        self.ui.tbl.setItem(row, BudgetColumn.BC_LABELS, QTableWidgetItem(title))
        font = self.ui.tbl.item(row, BudgetColumn.BC_LABELS).font()
        font.setBold(True)
        self.ui.tbl.item(row, BudgetColumn.BC_LABELS).setFont(font)

        for package, flow in self._flow_budget.items():
            if package == 'meta':  # 'meta' is where we put the summary, totals, time, selected_cells_count etc.
                continue
            if (
                (title == self.SOURCES_SINKS and package == 'FLOW-JA-FACE')  # noqa W503
                or (title == self.SELECTED_ZONE and package != 'FLOW-JA-FACE')  # noqa W503
            ):  # noqa W503
                continue
            row = self._add_row()
            self.ui.tbl.setItem(row, BudgetColumn.BC_LABELS, QTableWidgetItem(package))
            self.ui.tbl.setItem(row, BudgetColumn.BC_IN, QTableWidgetItem(self._from_float(flow['in'])))
            self.ui.tbl.setItem(row, BudgetColumn.BC_OUT, QTableWidgetItem(self._from_float(flow['out'])))
        self._add_totals_row(title)

    def _add_sources_sinks(self) -> None:
        """Adds the budget data."""
        self._add_budgets(self.SOURCES_SINKS)

    def _add_selected_zone(self) -> None:
        """Adds the budget data."""
        self._add_budgets(self.SELECTED_ZONE)

    def _add_totals_row(self, title) -> None:
        """Adds the totals to the table."""
        totals = self._flow_budget['meta']['totals']
        row = self._add_row()
        self.ui.tbl.setItem(row, BudgetColumn.BC_LABELS, QTableWidgetItem('Total'))
        self.ui.tbl.setItem(row, BudgetColumn.BC_IN, QTableWidgetItem(self._from_float(totals[title]['total_in'])))
        self.ui.tbl.setItem(row, BudgetColumn.BC_OUT, QTableWidgetItem(self._from_float(totals[title]['total_out'])))
        font = self.ui.tbl.item(row, BudgetColumn.BC_LABELS).font()
        font.setBold(True)
        self.ui.tbl.item(row, BudgetColumn.BC_LABELS).setFont(font)
        self.ui.tbl.item(row, BudgetColumn.BC_IN).setFont(font)
        self.ui.tbl.item(row, BudgetColumn.BC_OUT).setFont(font)

    def _add_total_flow(self):
        """Adds the total flow row."""
        totals = self._flow_budget['meta']['totals']
        row = self._add_row()
        self.ui.tbl.setItem(row, BudgetColumn.BC_LABELS, QTableWidgetItem('Total Flow'))
        self.ui.tbl.setItem(row, BudgetColumn.BC_IN, QTableWidgetItem(self._from_float(totals['total_in'])))
        self.ui.tbl.setItem(row, BudgetColumn.BC_OUT, QTableWidgetItem(self._from_float(totals['total_out'])))
        font = self.ui.tbl.item(row, BudgetColumn.BC_LABELS).font()
        font.setBold(True)
        self.ui.tbl.item(row, BudgetColumn.BC_LABELS).setFont(font)
        self.ui.tbl.item(row, BudgetColumn.BC_IN).setFont(font)
        self.ui.tbl.item(row, BudgetColumn.BC_OUT).setFont(font)

    def _add_blank_row(self):
        self._add_row()

    def _add_summary(self):
        """Adds the summary stats to the table."""
        summary = self._flow_budget['meta']['summary']
        row = self._add_row()  # Summary heading row
        self.ui.tbl.setItem(row, BudgetColumn.BC_LABELS, QTableWidgetItem('Summary'))
        font = self.ui.tbl.item(row, BudgetColumn.BC_LABELS).font()
        font.setBold(True)
        self.ui.tbl.item(row, BudgetColumn.BC_LABELS).setFont(font)

        for label, value in summary.items():
            row = self._add_row()
            self.ui.tbl.setItem(row, BudgetColumn.BC_LABELS, QTableWidgetItem(label))
            self.ui.tbl.setItem(row, BudgetColumn.BC_IN, QTableWidgetItem(self._from_float(value)))

    def _add_row(self) -> int:
        """Inserts a row at the bottom and returns the index of the new last row."""
        self.ui.tbl.insertRow(self.ui.tbl.rowCount())
        return self.ui.tbl.rowCount() - 1

    def _size_rows_and_columns(self) -> None:
        """Sets the column widths."""
        self.ui.tbl.horizontalHeader().setSectionResizeMode(BudgetColumn.BC_LABELS, QHeaderView.ResizeToContents)
        self.ui.tbl.horizontalHeader().setSectionResizeMode(BudgetColumn.BC_IN, QHeaderView.Stretch)
        self.ui.tbl.horizontalHeader().setSectionResizeMode(BudgetColumn.BC_OUT, QHeaderView.Stretch)
        self.ui.tbl.resizeRowsToContents()
        self._make_row_heights_equal()

    def _make_read_only(self) -> None:
        """Makes the whole table read only."""
        for row in range(self.ui.tbl.rowCount()):
            for column in range(self.ui.tbl.columnCount()):
                if self.ui.tbl.item(row, column):
                    self.ui.tbl.item(row, column).setFlags(self.ui.tbl.item(row, column).flags() & ~Qt.ItemIsEditable)

    def _make_row_heights_equal(self) -> None:
        """resizeRowsToContents() leaves some rows higher than others (!!!?) so this is needed."""
        mn = sys.maxsize
        mx = ~sys.maxsize
        for row in range(self.ui.tbl.rowCount()):
            mn = min(mn, self.ui.tbl.rowHeight(row))
            mx = max(mx, self.ui.tbl.rowHeight(row))
        if mn != mx:
            min_acceptable = 10
            default_height = 24
            height = mn if mn >= min_acceptable else default_height
            for row in range(self.ui.tbl.rowCount()):
                self.ui.tbl.setRowHeight(row, height)
