"""UzfDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QSizePolicy
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.gui import options_util
from xms.mf6.gui.advanced_package_dialog import AdvancedPackageDialog
from xms.mf6.gui.options_gui import OptionsGui


class UzfDialog(AdvancedPackageDialog):
    """A dialog used for the UZF package."""
    def __init__(self, dlg_input, parent=None):
        """Initializes the class, sets up the ui, and loads the simulation.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, parent)

    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections, and self.default_sections should be set here.
        """
        self.sections = ['COMMENTS', 'OPTIONS', 'DIMENSIONS', 'DATA']
        self.default_sections = ['DATA']

    def setup_section(self, section_name):
        """Sets up a block of widgets.

        Args:
            section_name (str): name of the block
        """
        if section_name == 'DIMENSIONS':
            self.setup_dimensions_section()
        else:
            super().setup_section(section_name)

    def setup_dimensions_section(self):
        """Sets up the DIMENSIONS section."""
        name = 'DIMENSIONS'
        self.add_group_box_to_scroll_area(name)
        vlayout = self.uix[name]['layout']
        self._add_chk_spn('NTRAILWAVES', 'Number of trailing waves', self.dlg_input.data.ntrailwaves, vlayout)
        self._add_chk_spn('NWAVESETS', 'Number of wave sets', self.dlg_input.data.nwavesets, vlayout)

        # set size policy for group box so it only expands horizontally
        self.uix[name]['group'].setSizePolicy(QSizePolicy(QSizePolicy.Expanding, QSizePolicy.Maximum))

        # self.enable_layout_items(self.uix[name]['layout'], not self.dlg_input.locked)

    def _add_chk_spn(self, name, brief, default, vlayout):
        """Adds a checkbox and spin box to the layout.

        Args:
            name (str): The name of the option.
            brief (str): Short description.
            default (int): The default value for the spin box.
            vlayout (QLayout): The layout to add the widgets to.
        """
        # Reuse some code from options_util
        options_util.line_edit(name, brief, self.uix, 'int', default, True, vlayout, True, self.dlg_input.locked)

    # @overrides
    def setup_options(self, vlayout):
        """Sets up the options section, which is defined dynamically, not in the ui file.

        Args:
            vlayout (QVBoxLayout): The layout that the option widgets will be added to.
        """
        self.options_gui = OptionsGui(self)
        self.options_gui.setup(vlayout)

    def change_columns(self, use_aux):
        """Changes a table's columns given the list of auxiliary variables.

        Args:
            use_aux (bool): True to include AUXILIARY variables.
        """
        table_widget = self.get_table_widget('PACKAGEDATA')
        if table_widget:
            table_widget.change_columns('PACKAGEDATA', use_aux=use_aux)

    # @overrides
    def _get_sp_widget(self):
        """Returns the PeriodListWidget.

        Returns:
            (PeriodListWidget): See description.
        """
        return self.data_tabs_widget.ui_sp_list_widget

    @override
    def widgets_to_data(self) -> None:
        """Set dlg_input.data from widgets."""
        super().widgets_to_data()
        if not self.dlg_input.locked:
            self.dlg_input.data.ntrailwaves = self.uix['spn_ntrailwaves'].value()
            self.dlg_input.data.nwavesets = self.uix['spn_nwavesets'].value()
