"""FlowModelDataFinder class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.gui.gwt.flow_package_data_finder import FlowPackageDataFinder


class FlowModelDataFinder(FlowPackageDataFinder):
    """Helper class to raise dialog so user can select flow model, then search flow model for data."""
    def __init__(self, flow_ftype, dlg_input, parent=None):
        """Initializes the class.

        Args:
            flow_ftype (str): The file type used in the model name file (e.g. 'WEL6')
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(flow_ftype, dlg_input, parent)

    @override
    def _extract_data(self, data):
        """Searches flow model for packages with aux variables and returns a dict for the SSM SOURCES block.

        Searches the flow model for packages with aux variables named 'CONCENTRATION'.

        Args:
            data (GwfData): The flow model.

        Returns:
            (dict): The pname, srctype, and auxname lists.
        """
        flow_package_data = {'pnames': [], 'srctypes': [], 'auxnames': []}
        parent_model_ftype = self.dlg_input.data.model.ftype
        auxname_set = {'CONCENTRATION'} if parent_model_ftype == 'GWT' else {'TEMPERATURE'}
        auxmixed_set = {'EVT6', 'RCH6'}
        for package in data.packages:
            auxiliary_variables = package.options_block.get('AUXILIARY', [])
            for variable in auxiliary_variables:
                if variable.upper() in auxname_set:
                    flow_package_data['pnames'].append(package.pname)
                    flow_package_data['srctypes'].append('AUXMIXED' if package.ftype in auxmixed_set else 'AUX')
                    flow_package_data['auxnames'].append(variable)
        return flow_package_data
