"""FmiDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
import pandas as pd
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QCheckBox, QDialog, QLineEdit, QPushButton
from typing_extensions import override

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import SimulationItem
from xms.api.tree import tree_util
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg

# 4. Local modules
from xms.mf6.file_io import io_factory
from xms.mf6.gui import gui_util, options_util
from xms.mf6.gui.options_gui import OptionsGui
from xms.mf6.gui.package_dialog_base import PackageDialogBase
from xms.mf6.gui.widgets.list_block_table_widget import ListBlockTableWidget
from xms.mf6.misc.settings import Settings


class FmiDialog(PackageDialogBase):
    """The FMI package dialog."""
    def __init__(self, dlg_input, parent=None):
        """Initializes the dialog.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, parent)

        self.flow_sim_uuid = ''  # UUID of the flow simulation

        self.setup_ui()

    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections, and self.default_sections should be set here.
        """
        self.sections = ['COMMENTS', 'OPTIONS', 'PACKAGEDATA']
        self.default_sections = ['PACKAGEDATA']

    def setup_section(self, section_name):
        """Sets up a section of widgets.

        Args:
            section_name (str): name of the section
        """
        if section_name == 'PACKAGEDATA':
            self.setup_packagedata_section()
        else:
            super().setup_section(section_name)

    # @overrides
    def setup_options(self, vlayout):
        """Sets up the options section, which is defined dynamically, not in the ui file.

        Args:
            vlayout (QVBoxLayout): The layout that the option widgets will be added to.
        """
        self.options_gui = OptionsGui(self)
        self.options_gui.setup(vlayout)

    def setup_packagedata_section(self):
        """Sets up the PACKAGEDATA section."""
        section = 'PACKAGEDATA'
        group_box = self.add_group_box_to_scroll_area(section)

        # GMS option
        check_str = 'Let GMS determine the paths from the flow simulation when saving'
        chk = self.uix[section]['chk_flow_sim'] = QCheckBox(check_str)
        self.uix[section]['layout'].addWidget(chk)
        chk.stateChanged.connect(self.on_chk_flow_sim)

        hlay_widget = options_util.create_hlayout_widget('flow_sim', self.uix[section])
        w = self.uix[section]['btn_flow_sim'] = QPushButton('Select Flow Simulation...')
        hlay_widget.layout().addWidget(w)
        w.clicked.connect(self.on_btn_flow_simulation)
        self.uix[section]['layout'].addWidget(hlay_widget)
        w = self.uix[section]['edt_flow_sim'] = QLineEdit()
        hlay_widget.layout().addWidget(w)
        w.setReadOnly(True)
        self.init_gms_option()

        # Table
        w = self.uix[section]['tbl_wgt'] = ListBlockTableWidget(self, self.dlg_input, '')
        self.uix[section]['layout'].addWidget(w)

        self.get_table_widget(section).load_list_block(section)
        self.uix[section]['tbl_wgt'].table.setColumnHidden(1, True)  # Hide FILEIN column

        group_box.setEnabled(not self.dlg_input.locked)
        chk.stateChanged.emit(chk.checkState())  # Manually emit this signal so dimming happens

    def on_chk_flow_sim(self, state):
        """Enables/disables the flow simulation stuff based on the checkbox.

        Args:
            state (int): The state of the check box.
        """
        section = 'PACKAGEDATA'
        if 'tbl_wgt' not in self.uix[section]:  # Abort if we haven't set up the PACKAGEDATA section yet
            return

        self.uix[section]['edt_flow_sim'].setEnabled(state == Qt.Checked)
        self.uix[section]['btn_flow_sim'].setEnabled(state == Qt.Checked)
        self.uix[section]['tbl_wgt'].setEnabled(state != Qt.Checked)

    def init_gms_option(self):
        """Populates the simulation line edit by attempting to get the flow simulation uuid from the settings."""
        gms_manages_paths = Settings.get(self.dlg_input.data.filename, 'GMS_MANAGES_PATHS')
        if gms_manages_paths == 'True':
            self.uix['PACKAGEDATA']['chk_flow_sim'].setChecked(True)

        self.flow_sim_uuid = Settings.get(self.dlg_input.data.filename, 'FLOW_SIMULATION_UUID')
        if self.flow_sim_uuid:
            sim_node = self.dlg_input.query.item_with_uuid(self.flow_sim_uuid)
            if not sim_node:
                self.flow_sim_uuid = ''
        self.update_flow_sim_path()

    def update_flow_sim_path(self):
        """Populates the simulation line edit."""
        if self.flow_sim_uuid:
            path = tree_util.build_tree_path(self.dlg_input.query.project_tree, self.flow_sim_uuid)
            self.uix['PACKAGEDATA']['edt_flow_sim'].setText(path)
        else:
            self.uix['PACKAGEDATA']['edt_flow_sim'].setText('<NOT DEFINED>')

    def on_btn_flow_simulation(self):
        """Called when the Select Flow Simulation button is pressed."""
        dialog = TreeItemSelectorDlg(
            title='Select Flow Simulation',
            target_type=SimulationItem,
            pe_tree=self.dlg_input.query.project_tree,
            override_icon=gui_util.get_icon_path,
            parent=self,
            allow_multi_select=False
        )
        if dialog.exec() == QDialog.Accepted:
            self.flow_sim_uuid = dialog.get_selected_item_uuid()
            self.update_selected_flow_model()

    def update_selected_flow_model(self):
        """Updates to the currently selected flow model."""
        sim_node = tree_util.find_tree_node_by_uuid(self.dlg_input.query.project_tree, self.flow_sim_uuid)
        if sim_node:
            self.flow_sim_uuid = sim_node.uuid
            sim_hidden = self.dlg_input.query.item_with_uuid(
                item_uuid=sim_node.uuid, model_name="MODFLOW 6", unique_name="Sim_Manager"
            )
            # Read the flow model
            mfsim_nam = sim_hidden.main_file
            reader = io_factory.reader_from_ftype('MFSIM6')
            mfsim = reader.read(mfsim_nam, sim_node, self.dlg_input.query)

            # Find the data and update the model
            columns = self._head_and_budget_files_from_sim(mfsim, sim_node.name)
            if not columns['FLOWTYPE']:
                message_box.message_with_ok(parent=self, message='No GWF flow model found.')
            else:
                model = self.get_table_widget('PACKAGEDATA').get_model('PACKAGEDATA')
                model.data_frame = pd.DataFrame(columns)
                model.data_frame.index += 1
                self.get_table_widget('PACKAGEDATA').set_model(model, 'PACKAGEDATA')
        else:
            self.flow_sim_uuid = ''
        self.update_flow_sim_path()

    @staticmethod
    def _add_file_from_options(package, model, sim_name, first_word, flowtype, columns):
        """Adds the head or budget file (if found) for the package to the columns.

        Args:
            package: Package data.
            model: The model.
            sim_name (str): Name of the flow simulation.
            first_word (str): 'HEAD' of 'HEAD FILEIN' or 'BUDGET' of 'BUDGET FILEIN'
            flowtype (str): 'GWFBUDGET', 'GWFHEAD' etc.
            columns (dict(str)): PACKAGEDATA block data.
        """
        file = package.options_block.get(f'{first_word} FILEOUT')
        if file:
            # rel_path = fs.compute_relative_path(mfsim_nam, file)
            rel_path = os.path.join('..', sim_name, f'{model.mname}_output', os.path.basename(file))
            FmiDialog._add_file(flowtype, rel_path, columns)

    @staticmethod
    def _add_file(flowtype, fname, columns):
        """Adds the info to the dict.

        Args:
            flowtype (str): 'GWFBUDGET', 'GWFHEAD' etc.
            fname (str): Relative path to the head or budget file.
            columns (dict(str)): PACKAGEDATA block data.
        """
        columns['FLOWTYPE'].append(flowtype)
        columns['FILEIN'].append('FILEIN')
        columns['FNAME'].append(fname)

    @staticmethod
    def _head_and_budget_files_from_sim(mfsim, sim_name):
        """Searches the simulation for head and budget files and returns a list of tuples.

        Args:
            mfsim (MfsimData): The simulation.
            sim_name (str): Name of the simulation.

        Returns:
            dict(str, str): List of tuples for the PACKAGEDATA block.
        """
        columns = {'FLOWTYPE': [], 'FILEIN': [], 'FNAME': []}
        ftype_set = {'OC6', 'MVR6', 'LAK6', 'MAW6', 'SFR6', 'UZF6'}
        for model in mfsim.models:
            if model.ftype == 'GWF6':
                for package in model.packages:
                    if package.ftype in ftype_set:
                        if package.ftype == 'OC6':
                            FmiDialog._add_file_from_options(package, model, sim_name, 'BUDGET', 'GWFBUDGET', columns)
                            FmiDialog._add_file_from_options(package, model, sim_name, 'HEAD', 'GWFHEAD', columns)
                        elif package.ftype == 'MVR6':
                            FmiDialog._add_file_from_options(package, model, sim_name, 'BUDGET', 'GWFMOVER', columns)
                        else:
                            FmiDialog._add_file_from_options(package, model, sim_name, 'BUDGET', package.pname, columns)
        return columns

    def save_flow_sim_option(self):
        """Saves the flow simulation uuid to the settings."""
        if self.uix['PACKAGEDATA']['chk_flow_sim'].isChecked() and self.flow_sim_uuid:
            Settings.set(self.dlg_input.data.filename, 'GMS_MANAGES_PATHS', 'True')
        else:
            Settings.set(self.dlg_input.data.filename, 'GMS_MANAGES_PATHS', 'False')

        Settings.set(self.dlg_input.data.filename, 'FLOW_SIMULATION_UUID', self.flow_sim_uuid)

    # @overrides
    def get_table_widget(self, block):
        """Returns the table view from the given block.

        Args:
            block: The block.
        """
        return self.uix['PACKAGEDATA']['tbl_wgt']

    @override
    def widgets_to_data(self) -> None:
        """Set dlg_input.data from widgets."""
        super().widgets_to_data()
        if not self.dlg_input.locked:
            self.save_flow_sim_option()
            self.save_list_blocks_to_temp()
