"""IcDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QCheckBox
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.ic_data import IcData
from xms.mf6.gui import gui_util
from xms.mf6.gui.griddata_dialog import GriddataDialog


class IcDialog(GriddataDialog):
    """A dialog used for the IC package."""
    def __init__(self, dlg_input, parent=None):
        """Initializes the class, sets up the ui, and loads the package.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, read_only_tabs=None, parent=parent)
        self._init_strt_equals_tops_option()

    @override
    def clear_sections(self) -> None:
        """Delete all section widgets."""
        self._init_strt_equals_tops_option()

    # @overrides
    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections, and self.default_sections should be set here.
        """
        if self.dlg_input.data.model.ftype == 'GWF6':
            self.sections = ['GMS Options', 'COMMENTS', 'GRIDDATA']
            self.default_sections = ['GMS Options', 'GRIDDATA']
            self.blocks_requiring_spacer.extend(['GMS Options'])
        else:  # 'STRT equals tops' option is only available for GWF6
            self.sections = ['COMMENTS', 'GRIDDATA']
            self.default_sections = ['GRIDDATA']

    def setup_section(self, section_name):
        """Sets up a block of widgets.

        Args:
            section_name (str): name of the block
        """
        if section_name == 'GMS Options':
            self._setup_gms_options_section()
        else:
            super().setup_section(section_name)

    def _setup_gms_options_section(self):
        """Sets up the GMS Options section."""
        name = 'GMS Options'
        self.add_group_box_to_scroll_area(name)
        self._setup_strt_equals_tops_option()

    def _setup_strt_equals_tops_option(self):
        """Sets up the GMS option 'Set starting heads equal to cell top elevations'."""
        name = 'GMS Options'
        str = 'Set starting heads equal to cell top elevations'
        chk = self.uix[name]['chk_strt_equals_tops'] = QCheckBox(str)
        self.uix[name]['layout'].addWidget(chk)
        chk.stateChanged.connect(self._on_chk_strt_equals_tops)
        self.enable_layout_items(self.uix[name]['layout'], not self.dlg_input.locked)

    def _strt_equals_tops(self) -> bool:
        """Returns True if the 'STRT equals tops' checkbox is checked.

        Returns:
            (bool): See description.
        """
        if 'GMS Options' in self.uix and 'chk_strt_equals_tops' in self.uix['GMS Options']:
            return self.uix['GMS Options']['chk_strt_equals_tops'].isChecked()
        return False

    def _on_chk_strt_equals_tops(self, state):
        """Enables/disables the STRT tab based on the checkbox.

        Args:
            state (int): The state of the check box.
        """
        strt_tab_index = gui_util.tab_index_from_text(self.griddata_widget.uix['tab_widget'], 'STRT')
        enabled = (state != Qt.Checked)
        self.griddata_widget.uix['tab_widget'].setTabEnabled(strt_tab_index, enabled)

    def _init_strt_equals_tops_option(self):
        """Sets the 'STRT equals tops' checkbox state."""
        if self.dlg_input.data.model.ftype == 'GWF6':
            strt_equals_tops = self.dlg_input.data.gms_options.get(IcData.STRT_EQUALS_TOPS, False)
            if strt_equals_tops:
                self.uix['GMS Options']['chk_strt_equals_tops'].setChecked(True)

    def _save_strt_equals_tops_option(self):
        """Saves the 'STRT equals tops' checkbox state."""
        self.dlg_input.data.gms_options[IcData.STRT_EQUALS_TOPS] = self._strt_equals_tops()

    @override
    def widgets_to_data(self) -> None:
        """Set dlg_input.data from widgets."""
        super().widgets_to_data()
        if not self.dlg_input.locked:
            self._save_strt_equals_tops_option()
