"""ListDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from functools import partial
import os
import shutil
import webbrowser

# 2. Third party modules
from PySide2.QtGui import QKeySequence
from PySide2.QtWidgets import (
    QAbstractItemView, QApplication, QDialogButtonBox, QHeaderView, QPushButton, QTableWidget, QTableWidgetItem,
    QToolBar, QVBoxLayout
)

# 3. Aquaveo modules
from xms.guipy.dialogs import choose_text_editor_dialog, message_box
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.testing import testing_tools
from xms.guipy.widgets import widget_builder

# 4. Local modules
from xms.mf6.components.default_package_creator import DefaultPackageCreator
from xms.mf6.data.grid_info import DisEnum
from xms.mf6.file_io import io_util
from xms.mf6.gui import gui_util
from xms.mf6.gui.resources.svgs import ROW_ADD_SVG, ROW_DELETE_SVG, ROW_DOWN_SVG, ROW_INSERT_SVG, ROW_UP_SVG
from xms.mf6.misc import util


def run_list_dialog_for_filein(ftype, parent, options_block, edit_func=None):
    """Opens the dialog and updates the list of files.

    This is a free function because of the second answer on https://stackoverflow.com/questions/35819538/

    Args:
        ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
        parent (Something derived from QWidget): The parent window.
        options_block (OptionsBlock): The OptionsBlock.
        edit_func: Function to call when Edit button is clicked.
    """
    option = f'{ftype} FILEIN'
    title = f'{option} Files'
    the_list = options_block.get(option, [])
    try:
        dialog = ListDialog(
            dialog_title=title,
            the_list=the_list,
            filename=parent.dlg_input.data.filename,
            mfsim_dir=parent.dlg_input.data.mfsim.dir_(),
            base_file_data=parent.dlg_input.data,
            edit_func=edit_func,
            ftype=ftype,
            locked=parent.dlg_input.locked,
            parent=parent,
            query=parent.dlg_input.query
        )
        # if dialog.exec() == QDialog.Accepted:
        #     move_files(options_block, option, dialog.the_list, parent.dlg_input.data.filename)
        # shutil.rmtree(dialog.temp_dir)
        dialog.exec()
        options_block.set(option, True, dialog.the_list)  # Can't cancel changes made to this. It's too hard.
    except Exception as error:
        raise error


# def move_files(options_block, option, the_list, component_file):
#     """Moves the files from the temp directory to the component folder.
#
#     Args:
#         options_block (OptionsBlock): package options.
#         option (str): The option.
#         the_list (list[str]): List of file names.
#         component_file (str): Filepath of component main file.
#     """
#     original_list = options_block.get(option, [])
#
#     # Delete original files
#     for filepath in original_list:
#         fs.removefile(filepath)
#
#     # Move files from temp dir to component folder
#     component_folder = os.path.normpath(os.path.dirname(component_file))
#     new_list = []
#     for filepath in the_list:
#         new_path = os.path.normpath(os.path.join(component_folder, os.path.basename(filepath)))
#         shutil.move(filepath, new_path)
#         new_list.append(new_path)
#
#     options_block.set(option, True, new_list)


class ListDialog(XmsDlg):
    """A dialog showing a simple list. Used for AUX variables and files."""

    text_column = 0
    """The column in the table for the text"""
    edit_column = 1
    """The column in the table for the Edit buttons"""
    uuid_column = 2
    """The column in the table for the uuids"""

    min_columns = 1
    """The maximum number of columns possible."""
    max_columns = 3
    """The maximum number of columns possible."""
    def __init__(
        self,
        dialog_title,
        the_list,
        filename='',
        mfsim_dir='',
        base_file_data=None,
        edit_func=None,
        ftype='',
        locked=False,
        parent=None,
        query=None
    ):
        """Initializes the dialog.

        Args:
            dialog_title (str): The text to show in the dialog window title.
            the_list (list of str): The list of strings.
            filename (str, optional): File path to package file so we can do
             relative paths. If empty, the edit button is not shown. AUX will not have a filename, but all FILEIN
             options will.
            mfsim_dir (str, optional): Path of the mfsim.nam directory.
            base_file_data (BaseFileData): Base file data object.
            edit_func: Function to call when Edit button is clicked.
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            locked (bool): True if the component is locked (read only).
            parent (Something derived from QWidget): The parent window.
            query (xmsapi.dmi.Query): Object for communicating with GMS
        """
        super().__init__(parent, 'xms.mf6.gui.list_dialog')

        self.the_list = the_list
        self.filename = filename
        self.mfsim_dir = mfsim_dir
        self.base_file_data = base_file_data
        self.edit_func = edit_func
        self.ftype = ftype
        self._locked = locked
        self.query = query
        self.actions = {}
        self._editing = False
        # self.temp_dir = ''

        self.setWindowTitle(dialog_title)
        # xms_parent_dlg.add_process_id_to_window_title(self)

        vertical_layout = QVBoxLayout(self)
        self._create_table_widget(vertical_layout)
        self._create_context_menu()
        # if self.filename:
        #     self._copy_files_to_temp_dir()
        self._add_items()
        self._create_toolbar(vertical_layout)
        self._create_button_box(vertical_layout)

        self.table_widget.resizeColumnToContents(ListDialog.edit_column)
        self.table_widget.setEnabled(not self._locked)

        # Signals
        self.table_widget.selectionModel().selectionChanged.connect(self.on_selection_changed)
        self.table_widget.itemChanged.connect(self.on_item_changed)
        self.btn_box.helpRequested.connect(self.help_requested)

        self.on_selection_changed()

    def _create_table_widget(self, vertical_layout):
        """Creates the table widget.

        Args:
            vertical_layout: The layout.
        """
        column_count = ListDialog.max_columns if self.filename else ListDialog.min_columns
        self.table_widget = QTableWidget(len(self.the_list), column_count, self)
        self.table_widget.horizontalHeader().hide()
        horz_header = self.table_widget.horizontalHeader()
        horz_header.setSectionResizeMode(0, QHeaderView.Stretch)
        if column_count == ListDialog.max_columns:
            self.table_widget.setColumnHidden(ListDialog.uuid_column, True)
        vertical_layout.addWidget(self.table_widget)
        self.table_widget.setSelectionMode(QAbstractItemView.SingleSelection)
        self.table_widget.verticalHeader().hide()

    # def _copy_files_to_temp_dir(self):
    #     """Copy all the files to a temporary area so the user can add/delete/edit but still cancel."""
    #     self.temp_dir = io_util.get_temp_filename()
    #     os.mkdir(self.temp_dir)
    #     the_list = []
    #     for filepath in self.the_list:
    #         newpath = os.path.join(self.temp_dir, os.path.basename(filepath))
    #         fs.copyfile(filepath, newpath)
    #         the_list.append(newpath)
    #     self.the_list = the_list

    def _add_items(self):
        """Adds the list to the rows of the table."""
        for row, item in enumerate(self.the_list):
            if self.filename:
                self.add_items_to_row(row, os.path.basename(item))
            else:
                self.add_items_to_row(row, item)

    def _create_context_menu(self):
        """Creates a context menu."""
        menu = [
            [ROW_INSERT_SVG, 'Insert', self.on_btn_insert],
            [ROW_DELETE_SVG, 'Delete', self.on_btn_delete],
            # ['copy', 'Copy', self.on_copy],
            # ['paste', 'Paste', self.on_paste]
        ]
        widget_builder.setup_context_menu(self.table_widget, menu)

    def _create_toolbar(self, vertical_layout):
        """Creates the toolbar.

        Args:
            vertical_layout: The layout
        """
        self.toolbar = QToolBar(self)
        button_list = [
            [ROW_INSERT_SVG, 'Insert Row', self.on_btn_insert], [ROW_ADD_SVG, 'Add Row', self.on_btn_add],
            [ROW_DELETE_SVG, 'Delete Row', self.on_btn_delete], [ROW_UP_SVG, 'Move Up', self.on_btn_up],
            [ROW_DOWN_SVG, 'Move Down', self.on_btn_down]
        ]
        self.actions = widget_builder.setup_toolbar(self.toolbar, button_list)
        vertical_layout.addWidget(self.toolbar)

    def _create_button_box(self, vertical_layout):
        """Creates the button box.

        Args:
            vertical_layout: The layout
        """
        self.btn_box = QDialogButtonBox()
        # self.btn_box.setStandardButtons(QDialogButtonBox.Cancel | QDialogButtonBox.Ok | QDialogButtonBox.Help)
        self.btn_box.setStandardButtons(QDialogButtonBox.Ok | QDialogButtonBox.Help)
        self.btn_box.accepted.connect(self.accept)
        self.btn_box.rejected.connect(self.reject)
        vertical_layout.addWidget(self.btn_box)

    def add_items_to_row(self, row, text):
        """Adds the widgets to the row.

        Args:
            row (int): The row (0-based).
            text (str): The file.
        """
        self._editing = True

        # Set the text column
        widget = QTableWidgetItem(text.strip('\'"'))
        self.table_widget.setItem(row, ListDialog.text_column, widget)

        if self.filename:
            # Set the uuid column
            uuid_str = testing_tools.new_uuid()
            widget = QTableWidgetItem(uuid_str)
            self.table_widget.setItem(row, ListDialog.uuid_column, widget)

            # Add the Edit... button
            margin = 15
            edit_text = 'Edit...'
            btn_edit = QPushButton(self.table_widget)
            btn_edit.setText(edit_text)
            width = btn_edit.fontMetrics().boundingRect(edit_text).width() + margin
            btn_edit.setMaximumWidth(width)
            btn_edit.clicked.connect(partial(self.on_btn_edit, uuid_str))
            self.table_widget.setCellWidget(row, ListDialog.edit_column, btn_edit)

        self._editing = False

    def on_selection_changed(self):
        """Called when the user clicks in the table and changes what cells are selected."""
        selection_list = self.table_widget.selectedIndexes()
        selections_exist = len(selection_list) > 0
        empty_table = (self.table_widget.rowCount() == 0)
        not_locked = not self._locked

        s = ':/resources/icons/'  # just to make the lines below shorter
        self.toolbar.widgetForAction(self.actions[f'{s}row-insert.svg']).setEnabled(not_locked and selections_exist)
        enabled = not_locked and (selections_exist or empty_table)
        self.toolbar.widgetForAction(self.actions[f'{s}row-add.svg']).setEnabled(enabled)
        self.toolbar.widgetForAction(self.actions[f'{s}row-delete.svg']).setEnabled(not_locked and selections_exist)
        self.toolbar.widgetForAction(self.actions[f'{s}row-up.svg']).setEnabled(not_locked and selections_exist)
        self.toolbar.widgetForAction(self.actions[f'{s}row-down.svg']).setEnabled(not_locked and selections_exist)

    def on_item_changed(self, item):
        """Called when the filename is edited."""
        if self.filename and not self._editing:
            # Get original file path
            row = item.row()
            original = self.the_list[row]

            # Get new file path and rename
            text = item.text()
            # new_path = os.path.join(self.temp_dir, text)
            new_path = os.path.join(os.path.dirname(original), text)
            shutil.move(original, new_path)
            self.the_list[row] = new_path

    def add_or_insert(self, add):
        """Called when the Add or Insert buttons are clicked. Adds rows to the table.

        Args:
            add (bool): True if adding, False if inserting
        """
        unique_rows = gui_util.get_unique_selected_rows(self.table_widget.selectedIndexes())
        row = self.table_widget.rowCount() if not unique_rows else min(unique_rows)
        if add and row < self.table_widget.rowCount():
            row = row + 1
        self.table_widget.insertRow(row)

        if self.filename:

            # Create the new file
            creator = DefaultPackageCreator()
            parent = self.base_file_data.tree_node.parent
            dis_enum = self.base_file_data.grid_info().dis_enum if self.base_file_data.grid_info() else DisEnum.DIS
            temp_filename = io_util.get_temp_filename(os.path.dirname(self.filename))
            new_filepath = creator.create_package(
                self.mfsim_dir, temp_filename, self.ftype, parent.name, dis_enum, overwrite=False
            )

            self.the_list.insert(row, new_filepath)
            self.add_items_to_row(row, os.path.basename(new_filepath))

            self.table_widget.resizeColumnToContents(ListDialog.edit_column)
        else:
            self.add_items_to_row(row, 'new')

    def on_btn_insert(self):
        """Called when the Insert button is clicked. Inserts rows to the spreadsheet."""
        self.add_or_insert(False)

    def on_btn_add(self):
        """Called when the Add button is clicked. Adds rows to the table."""
        self.add_or_insert(True)

    def on_btn_delete(self):
        """Called when the Delete button is clicked. Deletes rows from the table."""
        selected_list = self.table_widget.selectedIndexes()
        if not selected_list:
            message_box.message_with_ok(parent=self, message='Must select a row to delete.')
            return

        # Get the count of unique rows and the span from the selection
        unique_rows = gui_util.get_unique_selected_rows(selected_list)
        minrow = min(unique_rows)

        for row in range(minrow, minrow + 1):
            self.table_widget.removeRow(row)

        self.on_selection_changed()  # To update enabling

    def on_btn_up(self):
        """Called when the Up button is clicked. Moves rows up in the table."""
        selected_list = self.table_widget.selectedIndexes()
        if not selected_list:
            return

        self.move(True, selected_list)

    def on_btn_down(self):
        """Called when the Down button is clicked. Moves rows down in the table."""
        selected_list = self.table_widget.selectedIndexes()
        if not selected_list:
            return

        self.move(False, selected_list)

    def move(self, up, selected_list):
        """Moves a row up or down. From https://www.qtcentre.org/threads/3386-QTableWidget-move-row.

        Args:
            up (bool): True if moving up, else False
            selected_list (list of PySide2.QtCore.QModelIndex): The list of selected cells
        """
        self._editing = True
        source_row = selected_list[0].row()
        dest_row = source_row - 1 if up else source_row + 1
        if dest_row < 0 or dest_row > self.table_widget.rowCount() - 1:
            return

        # take whole rows
        source_items = self.take_row(source_row)
        dest_items = self.take_row(dest_row)

        # set back in reverse order
        self.set_row(source_row, dest_items)
        self.set_row(dest_row, source_items)

        self.table_widget.setCurrentCell(dest_row, 0)
        self._editing = True

    def take_row(self, row):
        """Calls QTableWidget.takeItem on the entire row.

        Args:
            row (int): The row (0-based)
        """
        row_items = []
        for col in range(self.table_widget.columnCount()):
            row_items.append(self.table_widget.takeItem(row, col))
        return row_items

    def set_row(self, row, row_items):
        """Calls QTableWidget.setItem on the entire row.

        Args:
            row (int): The row (0-based)
            row_items (list of items): The items in the row
        """
        for col in range(self.table_widget.columnCount()):
            self.table_widget.setItem(row, col, row_items[col])

    def _find_row(self, uuid):
        """Returns the row with the uuid in the uuid column.

        Args:
            uuid (str): A uuid.

        Returns:
            (int): The row index
        """
        for row in range(self.table_widget.rowCount()):
            item = self.table_widget.item(row, ListDialog.uuid_column)
            if item.text() == uuid:
                return row
        return -1

    def on_btn_edit(self, uuid_str: str) -> None:
        """Called when an Edit button is clicked.

        Args:
            uuid_str: The uuid of the row that was clicked.
        """
        filepath = self._filepath_from_uuid(uuid_str)
        if not os.path.exists(filepath):
            message_box.message_with_ok(parent=self, message='File not found.')
            return

        if self.edit_func:
            self.edit_func(filepath, self.base_file_data, self._locked, self)
        else:
            choose_text_editor_dialog.run_text_editor(file_path=filepath, win_cont=self, icon=util.get_app_icon())

    def _filepath_from_uuid(self, uuid_str) -> str:
        """Given a uuid corresponding to a particular row, return the filepath for that row.

        Args:
            uuid_str: Uuid for the row.

        Returns:
            See description.
        """
        row = self._find_row(uuid_str)
        item = self.table_widget.item(row, ListDialog.text_column)
        original = self.the_list[row]
        filepath = os.path.join(os.path.dirname(original), item.text())
        return filepath

    def copy_paste(self, do_copy):
        """Handles copy/paste.

        Args:
            do_copy (bool): True if copying, False if pasting.
        """
        selection_list = self.table_widget.selectedIndexes()
        if len(selection_list) > 0:
            item = self.table_widget.itemFromIndex(selection_list[0])
            if item:
                if do_copy:
                    QApplication.clipboard().setText(item.text())
                else:
                    text = QApplication.clipboard().text()
                    item.setText(text)

    def on_copy(self):
        """Handles copy."""
        self.copy_paste(True)

    def on_paste(self):
        """Handles paste."""
        self.copy_paste(False)

    def keyPressEvent(self, event):  # noqa N802 (function name 'keyPressEvent' should be lowercase)
        """Handle Copy/Paste via CTRL+C, CTRL+V.

        Args:
            event (QKeyEvent): The event
        """
        if event.matches(QKeySequence.Copy):
            self.on_copy()
        elif event.matches(QKeySequence.Paste):
            self.on_paste()

    def help_requested(self):
        """Opens a web browser to a URL when the user clicks the Help button."""
        page = ''
        if self.filename:
            if self.ftype == 'OBS6':
                page = 'MF6_Observation_Files_Dialog'
            elif self.ftype == 'TAS6':
                page = 'MF6_Time-Array_Series_Files_Dialog'
            elif self.ftype == 'TVA6':
                page = 'MF6_Time_Variable_Array_Files_Dialog'
            elif self.ftype == 'GNC6':
                page = 'MF6_Ghost_Node_Correction_Files_Dialog'
            elif self.ftype == 'MVR6':
                page = 'MF6_Water_Mover_Files_Dialog'
        else:
            page = 'Auxiliary_Variables_Dialog'
        url = f'{util.wiki_gms}:{page}'
        webbrowser.open(url)

    def accept(self):
        """Called when the OK button is clicked.

        Saves the table contents to self.the_list and closes the dialog.
        """
        if not self._locked:
            the_list = []
            for row in range(self.table_widget.rowCount()):
                item = self.table_widget.item(row, ListDialog.text_column)
                if not self.filename:
                    the_list.append(item.text())
                else:
                    # the_list.append(os.path.join(self.temp_dir, item.text()))
                    original = self.the_list[row]
                    the_list.append(os.path.join(os.path.dirname(original), item.text()))
            self.the_list = the_list
        super().accept()

    def reject(self):
        """Called when the Cancel button is clicked."""
        super().reject()
