"""Functions for creating widgets in the OPTIONS blocks."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import (
    QCheckBox, QComboBox, QDoubleSpinBox, QHBoxLayout, QLabel, QLineEdit, QPushButton, QSizePolicy, QSpinBox, QWidget
)

# 3. Aquaveo modules
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.widgets import widget_builder

# 4. Local modules
from xms.mf6.gui.options_defs import Checkbox


def create_hlayout_widget(key, widget_dict) -> QWidget:
    """Creates and returns a widget with a QHBoxLayout.

    Useful if you need to disable the entire thing.

    Args:
        key (str): options key and name of the widgets.
        widget_dict (dict): Dict of options widgets.

    Returns:
        See description.
    """
    widget = QWidget()
    if widget_dict:
        widget_dict[f'widget_{key.lower()}'] = widget
    hlayout = QHBoxLayout()
    hlayout.setContentsMargins(0, 0, 0, 0)
    widget.setLayout(hlayout)
    return widget


def make_widget_name(widget_type_abbreviation, key):
    """Returns a string to be used as the name of the widget in a dict.

    Args:
        widget_type_abbreviation (str): 'chk', 'btn', 'cbx' etc.
        key (str): The key string from the options dict.
    """
    return f'{widget_type_abbreviation}_{key.lower().replace(" ", "_")}'


def checkbox_or_label(key, brief, widget_dict, vlayout, append_colon=False, whats_this='', label=False):
    """Sets up an options checkbox.

    Args:
        key (str): Options key and name of the widget.
        brief (str): Optional verbage that follows the key.
        widget_dict (dict): Dict of options widgets.
        vlayout (QVBoxLayout): The vertical layout that the option widgets will be added to.
        append_colon (bool): If True, appends a ':' to the end of the string so another widget can follow.
        whats_this (str): Optional string that will appear in a popup if user right-clicks.
        label (bool): If True, a label is created instead of a checkbox.

    Returns:
        (QCheckBox): The checkbox
    """
    checkbox_text = f'{key} - {brief}' if brief else f'{key}'
    checkbox_text = checkbox_text if not append_colon else f'{checkbox_text}:'
    if label:
        name = make_widget_name('txt', key)
        w = widget_dict[name] = QLabel(checkbox_text)
    else:
        name = make_widget_name('chk', key)
        w = widget_dict[name] = QCheckBox(checkbox_text)
    vlayout.addWidget(w)
    if whats_this:  # Here's an idea to use the '?' thing to give the user more info.
        w.setWhatsThis(whats_this)
    w.setObjectName(name)  # Useful for debugging
    return w


def checkbox_checkbox(key, brief, key2, brief2, widget_dict, vlayout):
    """Sets up a horizontal layout with a checkbox followed by a line edit.

    Args:
        key (str): Options key and name of the widget.
        brief (str): Optional verbage that follows the key.
        key2 (str): Options key and name of the widget.
        brief2 (str): Optional verbage that follows the key.
        widget_dict (dict): Dict of options widgets.
        vlayout (QVBoxLayout): The vertical layout that the option widgets will be added to.

    Returns:
        (tuple): tuple containing:
            - (QCheckBox): The first checkbox.
            - (QCheckBox): The second checkbox.
    """
    hlay_widget = create_hlayout_widget(key, widget_dict)
    vlayout.addWidget(hlay_widget)
    chk = checkbox_or_label(key, brief, widget_dict, hlay_widget.layout())
    chk2 = checkbox_or_label(key2, brief2, widget_dict, hlay_widget.layout())
    hlay_widget.layout().addStretch()
    return chk, chk2


def checkbox_button(key, brief, button_text, widget_dict, vlayout):
    """Sets up a horizontal layout with a checkbox followed by a line edit.

    Args:
        key (str): Options key and name of the widget.
        brief (str): Optional verbage that follows the key.
        button_text (str): Text on the button.
        widget_dict (dict): Dict of options widgets.
        vlayout (QVBoxLayout): The vertical layout that the option widgets will be added to.

    Returns:
        (tuple): tuple containing:
            - hlayout (QHBoxLayout): The horizontal layout that the widgets are in.
            - (QCheckBox): The first checkbox.
            - (QCheckBox): The second checkbox.
    """
    hlay_widget = create_hlayout_widget(key, widget_dict)
    vlayout.addWidget(hlay_widget)
    chk = checkbox_or_label(key, brief, widget_dict, hlay_widget.layout())
    btn = widget_dict[make_widget_name('btn', key)] = QPushButton(button_text)
    hlay_widget.layout().addWidget(btn)
    hlay_widget.layout().addStretch()
    return hlay_widget, chk, btn


def add_line_edit_or_spin(
    key, hlay_widget, widget_dict, type_, value, add_label, add_stretch, minimum=None, maximum=None
):
    """Adds a QLineEdit (if type_ == 'float' or 'str') or QSpinBox (if type_ == 'int') to the horizontal layout widget.

    We don't use a QDoubleSpinBox for floats because it hardwires the precision.

    Args:
        key (str): options key and name of the widgets
        hlay_widget (QWidget): The widget used to define the horizontal layout.
        widget_dict (dict): Dict of options widgets.
        type_ (str): 'str' for string, 'int' for int, 'float' for float
        value: Default value
        add_label (bool): If True, a label is added to the left of the field.
        add_stretch (bool): If True, a stretch is added as the last thing in the layout
        minimum: The minimum value.
        maximum: The maximum value.

    Returns:
        (QLineEdit or QSpinBox): The line edit.
    """
    if add_label:
        w = widget_dict[make_widget_name('txt', key)] = QLabel(f'{key}:')
        hlay_widget.layout().addWidget(w)

    if type_ == 'int':
        name = make_widget_name('spn', key)
        w = widget_dict[name] = QSpinBox()
        if minimum:
            w.setMinimum(minimum)
        if maximum:
            w.setMaximum(maximum)
        if value is not None:
            w.setValue(int(value))
        # These seem to default to be too wide. The following changes that.
        w.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.MinimumExpanding)
        w.setMaximumWidth(150)
    elif type_ == 'float':
        name = make_widget_name('edt', key)
        w = widget_dict[name] = QLineEdit()
        if value is not None:
            w.setText(str(value))
        validator = QxDoubleValidator(parent=w)
        if minimum:
            validator.setBottom(minimum)
        if maximum:
            validator.setTop(maximum)
        w.setValidator(validator)
    else:
        name = make_widget_name('edt', key)
        w = widget_dict[name] = QLineEdit()
        if 'FILEIN' in key or 'FILEOUT' in key:
            w.setAlignment(Qt.AlignHCenter)
            w.setPlaceholderText('file name')
        if value is not None:
            w.setText(str(value))

    w.setObjectName(name)  # Useful for debugging
    hlay_widget.layout().addWidget(w)
    if add_stretch:
        hlay_widget.layout().addStretch()
    return w


def checkbox_field(
    key, brief, widget_dict, type_, value, vlayout, add_stretch=True, read_only=False, minimum=None, maximum=None
):
    """Sets up a horizontal layout with a check box followed by a line edit or, if type_=='int', a spin box.

    We don't use a QDoubleSpinBox for floats because it hardwires the precision.

    Args:
        key (str): options key and name of the widgets
        brief (str): Optional verbage that follows the key.
        widget_dict (dict): Dict of options widgets.
        type_ (str): 'str' for string, 'int' for int, 'float' for float
        value: Default value
        vlayout (QVBoxLayout): The vertical layout that the option widgets will be added to.
        add_stretch (bool): If True, a stretch is added as the last thing in the layout
        read_only (bool): If True, checkbox and field are made read-only.
        minimum: The minimum value.
        maximum: The maximum value.

    Returns:
        (tuple): tuple containing:
            - (QWidget): Widget defining the layout.
            - (QCheckBox): The checkbox
            - The QLineEdit or QSpinBox
    """
    hlay_widget = create_hlayout_widget(key, widget_dict)
    vlayout.addWidget(hlay_widget)
    chk = checkbox_or_label(key, brief, widget_dict, hlay_widget.layout(), append_colon=True)
    field = add_line_edit_or_spin(
        key,
        hlay_widget,
        widget_dict,
        type_,
        value,
        add_label=False,
        add_stretch=add_stretch,
        minimum=minimum,
        maximum=maximum
    )
    if read_only:
        widget_builder.make_lineedit_readonly(field)
    return hlay_widget, chk, field


def line_edit(
    key,
    brief,
    widget_dict,
    type_,
    value,
    label,
    vlayout,
    add_stretch=True,
    read_only=False,
    minimum=None,
    maximum=None
):
    """Sets up a horizontal layout with a label followed by a line edit.

    We don't use a QDoubleSpinBox for floats because it hardwires the precision.

    Args:
        key (str): options key and name of the widgets
        brief (str): Optional verbage that follows the key.
        widget_dict (dict): Dict of options widgets.
        type_ (str): 'str' for string, 'int' for int, 'float' for float
        value: Default value
        label (bool): If True, a label is created instead of a checkbox.
        vlayout (QVBoxLayout): The vertical layout that the option widgets will be added to.
        add_stretch (bool): If True, a stretch is added as the last thing in the layout
        read_only (bool): If True, checkbox and field are made read-only.
        minimum: The minimum value.
        maximum: The maximum value.

    Returns:
        (tuple): tuple containing:
            - (QWidget): Widget defining the layout.
            - (QCheckBox): The checkbox
            - The QLineEdit or QSpinBox
    """
    hlay_widget = create_hlayout_widget(key, widget_dict)
    vlayout.addWidget(hlay_widget)
    label = checkbox_or_label(key, brief, widget_dict, hlay_widget.layout(), append_colon=True, label=label)
    field = add_line_edit_or_spin(
        key,
        hlay_widget,
        widget_dict,
        type_,
        value,
        add_label=False,
        add_stretch=add_stretch,
        minimum=minimum,
        maximum=maximum
    )
    if read_only:
        widget_builder.make_lineedit_readonly(field)
    return hlay_widget, label, field


def setup_checkbox_line_edit_button(key, brief, type_, value, button_text, widget_dict, vlayout):
    """Sets up a horizontal layout with a checkbox followed by a line edit followed by a button.

    Args:
        key (str): options key and name of the widgets
        brief (str): Optional verbage that follows the key.
        type_ (str): 'str' for string, 'int' for int, 'float' for float
        value: Default value
        button_text (str): Text on the button.
        widget_dict (dict): Dict of options widgets.
        vlayout (QVBoxLayout): The vertical layout that the option widgets will be added to.

    Returns:
        (tuple): tuple containing:
            - (QWidget): Widget defining the layout.
            - (QCheckBox): The checkbox
            - The QLineEdit or QSpinBox
            - (QPushButton): The button
    """
    hlay_widget, chk, field = checkbox_field(key, brief, widget_dict, type_, value, vlayout, add_stretch=False)
    btn = widget_dict[make_widget_name('btn', key)] = QPushButton(button_text)
    hlay_widget.layout().addWidget(btn)
    hlay_widget.layout().addStretch()
    return hlay_widget, chk, field, btn


def checkbox_spinbox(key, brief, widget_dict, type_, value, min, max, vlayout, whats_this=''):
    """Sets up a horizontal layout with a checkbox followed by a line edit.

    Args:
        key (str): options key and name of the widgets
        brief (str): Optional verbage that follows the key.
        widget_dict (dict): Dict of options widgets.
        type_ (str): 'str' for string, 'int' for int, 'float' for float
        value (int or float): Initial value.
        min (int or float): Minimum value.
        max (int or float): Maximum value.
        vlayout (QVBoxLayout): The vertical layout that the option widgets will be added to.
        whats_this (str): Optional string that will appear in a popup if user right-clicks.

    Returns:
        (QCheckBox): The checkbox
    """
    hlay_widget = create_hlayout_widget(key, widget_dict)
    vlayout.addWidget(hlay_widget)

    chk = checkbox_or_label(key, brief, widget_dict, hlay_widget.layout(), append_colon=True, whats_this=whats_this)

    if type_ == 'int':
        w = widget_dict[f'spn_{key.lower()}'] = QSpinBox()
        # These seem to default to be too wide. The following changes that.
        w.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.MinimumExpanding)
        w.setMaximumWidth(100)
    elif type_ == 'float':
        w = widget_dict[f'spn_{key.lower()}'] = QDoubleSpinBox()
        # These seem to default to be too wide. The following changes that.
        w.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.MinimumExpanding)
        w.setMaximumWidth(100)
    else:
        raise RuntimeError(f'Unrecognized type {type} passed to checkbox_spinbox.')
    if min:
        w.setMinimum(min)
    if max:
        w.setMaximum(max)
    w.setValue(value)
    hlay_widget.layout().addWidget(w)
    hlay_widget.layout().addStretch()

    return chk


def checkbox_combobox(key, brief, widget_dict, items, vlayout):
    """Sets up a horizontal layout with a checkbox followed by a combo box.

    E.g. "CALC_METHOD - Calculation method:"
    key = "CALC_METHOD"
    brief = "Calculation method"

    Args:
        key (str): options key and name of the widgets
        brief (str): Optional verbage that follows the key.
        widget_dict (dict): Dict of options widgets.
        items (list(str)): The strings for the combo box.
        vlayout (QVBoxLayout): The vertical layout that the option widgets will be added to.

    Returns:
        (tuple): tuple containing:
            - hlayout (QHBoxLayout): The horizontal layout that the widgets are in.
            - (QCheckBox): The checkbox.
            - (QComboBox): The combobox.
    """
    hlay_widget = create_hlayout_widget(key, widget_dict)
    vlayout.addWidget(hlay_widget)
    chk = checkbox_or_label(key, brief, widget_dict, vlayout, append_colon=True)
    hlay_widget.layout().addWidget(chk)
    cbx = widget_dict[make_widget_name('cbx', key)] = QComboBox()
    hlay_widget.layout().addWidget(cbx)
    hlay_widget.layout().addStretch()
    cbx.addItems(items)
    return hlay_widget, chk, cbx


def setup_spinbox(key, widget_dict, hlay_widget, type_='int'):
    """Creates a label and spinbox.

    Args:
        key (str): options key and name of the widgets.
        hlay_widget (QWidget): A layout widget.
        widget_dict (dict): Dict of options widgets.
        type_ (str): 'str' for string, 'int' for int, 'float' for float

    Returns:
        A QWidget containing the child widgets.
    """
    w = widget_dict[f'txt_{key.lower()}'] = QLabel(f'{key}:')
    hlay_widget.layout().addWidget(w)
    if type_ == 'int':
        w = widget_dict[f'spn_{key.lower()}'] = QSpinBox()
        # These seem to default to be too wide. The following changes that.
        w.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.MinimumExpanding)
        w.setMaximumWidth(100)
    elif type_ == 'float':
        w = widget_dict[f'spn_{key.lower()}'] = QDoubleSpinBox()
        # These seem to default to be too wide. The following changes that.
        w.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.MinimumExpanding)
        w.setMaximumWidth(100)
    w.setMinimum(0)
    w.setMaximum(1e6)
    w.setValue(1)
    hlay_widget.layout().addWidget(w)
    return hlay_widget


def checkbox_3_fields(
    checkbox_key,
    key1,
    type1,
    value1,
    key2,
    type2,
    value2,
    key3,
    type3,
    value3,
    widget_dict,
    vlayout,
    minimum1=None,
    maximum1=None,
    minimum2=None,
    maximum2=None,
    minimum3=None,
    maximum3=None
):
    """Creates the widgets for the REWET option (and similar options in other packages).

    Args:
        checkbox_key (str): Checkbox option key.
        key1 (str): First edit field option key.
        type1 (str): First edit field type: 'str' for string, 'int' for int, 'float' for float
        value1: First default value.
        key2 (str): Second edit field option key.
        type2 (str): Second edit field type: 'str' for string, 'int' for int, 'float' for float
        value2: Second default value.
        key3 (str): Third edit field option key.
        type3 (str): Third edit field type: 'str' for string, 'int' for int, 'float' for float
        value3: Third default value.
        widget_dict (dict): Dict of options widgets.
        vlayout (QVBoxLayout): The vertical layout that the option widgets will be added to.
        minimum1: First edit field minimum value.
        maximum1: First edit field maximum value.
        minimum2: Second edit field minimum value.
        maximum2: Second edit field maximum value.
        minimum3: Third edit field minimum value.
        maximum3: Third edit field maximum value.

    Returns:
        (QCheckBox): The checkbox.
    """
    hlay1 = create_hlayout_widget(checkbox_key, widget_dict)
    vlayout.addWidget(hlay1)
    chk = checkbox_or_label(checkbox_key, '', widget_dict, vlayout)
    vlayout.addWidget(hlay1)

    hlay2 = create_hlayout_widget(key1, widget_dict)
    vlayout.addWidget(hlay2)
    widget1 = add_line_edit_or_spin(
        key1, hlay2, widget_dict, type1, value1, add_label=True, add_stretch=True, minimum=minimum1, maximum=maximum1
    )
    widget2 = add_line_edit_or_spin(
        key2, hlay2, widget_dict, type2, value2, add_label=True, add_stretch=True, minimum=minimum2, maximum=maximum2
    )
    widget3 = add_line_edit_or_spin(
        key3, hlay2, widget_dict, type3, value3, add_label=True, add_stretch=True, minimum=minimum3, maximum=maximum3
    )
    return chk, widget1, widget2, widget3


def setup_print_format(first_word, widget_dict, vlayout):
    """Creates the widgets for the HEAD PRINT_FORMAT option (and similar options in other packages).

    OC (GWF) = HEAD PRINT_FORMAT
    OC (GWT) = CONCENTRATION PRINT_FORMAT
    IST (GWT) = CIM PRINT_FORMAT

    Args:
        first_word (str): 'HEAD' for maw6, 'STAGE' for sfr6 and lak6 etc.
        widget_dict (dict): Dict of options widgets.
        vlayout (QVBoxLayout): The vertical layout that the option widgets will be added to.

    Returns:
        (QWidget): The layout widget containing all the widgets that were created.
    """
    hlay1, chk, cbx = checkbox_combobox(
        f'{first_word} PRINT_FORMAT', '', widget_dict, ['EXPONENTIAL', 'FIXED', 'GENERAL', 'SCIENTIFIC'], vlayout
    )
    vlayout.addWidget(hlay1)

    hlay2 = create_hlayout_widget(f'{first_word.lower()}_print_format', widget_dict)
    spn_columns = setup_spinbox('COLUMNS', widget_dict, hlay2)
    spn_width = setup_spinbox('WIDTH', widget_dict, hlay2)
    spn_digits = setup_spinbox('DIGITS', widget_dict, hlay2)
    hlay2.layout().addStretch()
    vlayout.addWidget(hlay2)
    return chk, cbx, spn_columns, spn_width, spn_digits


def load_print_format_options(first_word, options, widget_dict):
    """Loads the options into the PRINT_FORMAT widgets.

    Args:
        first_word (str): HEAD (OC GWF), CONCENTRATION (OC GWT), CIM (IST)
        options (dict): The options dict.
        widget_dict (dict): Dict of options widgets.
    """
    # Initialize to defaults
    columns = 10
    width = 15
    digits = 6
    format_ = 'GENERAL'

    print_format = options.get(f'{first_word} PRINT_FORMAT')
    widget_dict[f'chk_{first_word.lower()}_print_format'].setChecked(print_format is not None)
    if print_format is not None:
        words = print_format.split()
        columns = int(words[1]) if len(words) > 1 else columns
        width = int(words[3]) if len(words) > 3 else width
        digits = int(words[5]) if len(words) > 5 else digits
        format_ = words[6].upper() if len(words) > 6 else format_
    widget_dict['spn_columns'].setValue(columns)
    widget_dict['spn_width'].setValue(width)
    widget_dict['spn_digits'].setValue(digits)
    widget_dict[f'cbx_{first_word.lower()}_print_format'].setCurrentText(format_)


def save_print_format_options(first_word, options_block, widget_dict):
    """Saves the print format options from the widgets to the dict.

    Args:
        first_word (str): HEAD (OC GWF), CONCENTRATION (OC GWT), CIM (IST)
        options_block (OptionsBlock): The options block.
        widget_dict (dict): Dict of options widgets.
    """
    words = [
        'COLUMNS',
        str(widget_dict['spn_columns'].value()), 'WIDTH',
        str(widget_dict['spn_width'].value()), 'DIGITS',
        str(widget_dict['spn_digits'].value()), widget_dict[f'cbx_{first_word.lower()}_print_format'].currentText()
    ]
    options_block.set(
        f'{first_word} PRINT_FORMAT', widget_dict[f'chk_{first_word.lower()}_print_format'].isChecked(), words
    )


def is_areal_package(data):
    """Returns true if package ftype is 'EVT6' or 'RCH6'. Can be list or array based.

    Args:
        data: Something derived from BaseFileData.

    Returns:
        See description.
    """
    return data.ftype in ['EVT6', 'RCH6']


def export_array_ascii_def() -> Checkbox:
    """Returns the definition for the EXPORT_ARRAY_ASCII option, which is used a lot.

    Returns:
        See description.
    """
    export_array_ascii_brief = 'Input griddata arrays should be written to layered ascii output files'
    return Checkbox('EXPORT_ARRAY_ASCII', brief=export_array_ascii_brief)
