"""PestObsDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
from PySide2.QtGui import QTextCursor
from PySide2.QtWidgets import QApplication, QDialog

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.core.filesystem import filesystem as fs
from xms.data_objects.parameters import Coverage
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.mf6.file_io.pest import pest_obs_data_generator
from xms.mf6.file_io.pest.pest_obs_data_generator import PestObsDataGenerator
from xms.mf6.file_io.pest.pest_obs_util import CovInfo, ObsCovData
from xms.mf6.gui import gui_util
from xms.mf6.gui.dialog_input import DialogInput
from xms.mf6.gui.pest.generate_pest_obs_dialog import GeneratePestObsDialog
from xms.mf6.gui.pest.pest_obs_dialog_ui import Ui_PestObsDialog
from xms.mf6.mapping import map_util


class PestObsDialog(XmsDlg):
    """A dialog that appears when creating a new simulation."""
    def __init__(self, dlg_input, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(parent, 'xms.mf6.gui.pest_obs_dialog')
        self.dlg_input = dlg_input
        self._files = []
        self._current_row = 0
        self._current_original_text = ''  # Avoids having to read file from disk as they're editing
        self.help_getter = gui_util.help_getter(dlg_input.help_id)

        self.ui = Ui_PestObsDialog()
        self.ui.setupUi(self)

        # Signals
        self.ui.buttonBox.helpRequested.connect(self.help_requested)
        self.ui.btnGenerate.clicked.connect(self._on_btn_generate)
        self.ui.btnDeleteAll.clicked.connect(self._on_btn_delete_all)
        self.ui.btnSave.clicked.connect(self._on_btn_save)
        self.ui.lstFiles.currentRowChanged.connect(self._on_change_current_file)
        self.ui.edtFileContents.textChanged.connect(self._on_text_changed)

        # Handle splitter
        self._setup_splitter()
        QApplication.processEvents()
        self.adjustSize()

        self.ui.tabWidget.setCurrentIndex(0)
        self._load_files()
        gui_util.handle_read_only_notice(self.dlg_input.locked, self.ui.txt_read_only)
        self._lock_widgets()
        self._update_save_button_status()

    def _on_text_changed(self):
        self._update_save_button_status()

    def _on_btn_save(self):
        file_name = self._get_current_file()
        text = self.ui.edtFileContents.toPlainText()
        with open(file_name, 'w') as file:
            file.write(text)
        self._current_original_text = text
        self._update_save_button_status()

    def _file_has_changes(self):
        # file_name = self._get_current_file()
        # file_text = self._text_from_file(file_name)
        edt_text = self.ui.edtFileContents.toPlainText()
        # return edt_text != file_text
        return edt_text != self._current_original_text

    def _update_save_button_status(self):
        self.ui.btnSave.setEnabled(self._file_has_changes())

    def _lock_widgets(self):
        if self.dlg_input.locked:
            self.ui.btnGenerate.setEnabled(False)
            self.ui.btnDeleteAll.setEnabled(False)
            self.ui.btnSave.setEnabled(False)
            self.ui.edtFileContents.setReadOnly(True)

    def _load_files(self):
        folder = os.path.dirname(self.dlg_input.data.filename)
        extensions_set = {'.bsamp', '.n2b', '.blisting', '.b2b', '.fsamp'}
        for _, _, files in os.walk(folder):
            for f in files:
                extension = os.path.splitext(f)[1]
                if extension in extensions_set:
                    self._files.append(os.path.basename(f))
        self._setup_file_list(self._files)
        self._display_current_file()

    def _on_btn_generate(self):
        """Called when the generate button is clicked."""
        dep_var_cov_data, flow_cov_data = _get_obs_coverages(self.dlg_input)
        model_ftype = self.dlg_input.data.model.ftype
        dialog = GeneratePestObsDialog(model_ftype, dep_var_cov_data, flow_cov_data, self)
        if dialog.exec() == QDialog.Accepted:
            self._on_btn_delete_all()
            dep_var_cov_data = dialog.get_coverage_data('dep_var')
            flow_cov_data = dialog.get_coverage_data('flow')
            nearest = dialog.nearest_n_points()
            generator = PestObsDataGenerator(self.dlg_input.data)
            self._files, errors = generator.generate(dep_var_cov_data, flow_cov_data, nearest)
            if errors:
                self.ui.edtErrors.setPlainText(errors)
                message_box.message_with_ok(parent=self, message='There were errors.')
                self.ui.tabWidget.setCurrentIndex(1)
            self._setup_file_list(self._files)
            self._display_current_file()

    def _ask_to_save_if_changes(self):
        """Ask the user if they want to save their changes if they haven't yet and they are changing files."""
        if self._file_has_changes():
            rv = message_box.message_with_n_buttons(
                parent=self,
                message='Save the current changes?',
                button_list=['Save Changes', 'Discard Changes'],
                default=0,
                escape=1
            )
            if rv == 0:
                self._on_btn_save()

    def _on_change_current_file(self, current_row):
        """Called when the current file changes."""
        self._ask_to_save_if_changes()
        self._current_row = current_row
        self._display_current_file()
        self._update_description()
        self._update_save_button_status()

    def _get_current_file(self):
        """Returns the current filename."""
        # current_row = self.ui.lstFiles.currentRow()
        current_row = self._current_row
        if current_row >= 0 and self._files:
            return self._files[current_row]
        return ''

    def _update_description(self):
        """Updates the description of the file."""
        file_name = self._get_current_file()
        description = ''
        if file_name.endswith('.bsamp') or file_name.endswith('.fsamp'):
            description = (
                '"Bore Sample File". This file contains observed values at discrete times. A time of'
                ' 01/01/1950 is the default and is interpreted as 0 by PEST.'
            )
        elif file_name.endswith('.n2b'):
            description = (
                '"Node-to-Bore Interpolation File". This file is used to interpolate from cell centers to'
                ' the observation points. It shows computed values and interpolation weights for the N'
                ' closest cell centers to each point.'
            )
        elif file_name.endswith('.blisting'):
            description = ('"Bore Listing File". This file contains the names of the observation locations.')
        elif file_name.endswith('.b2b'):
            description = (
                '"Bore-to-Budget File". This file contains info on how to obtain computed flow from the'
                ' budget file.'
            )
        self.ui.txtDescription.setText(description)

    def _display_current_file(self):
        self.ui.edtFileContents.clear()
        file_name = self._get_current_file()
        if file_name:
            self._fill_edt_with_file(file_name, self.ui.edtFileContents)
            # This code uses a shared function that replaces spaces with tabs but it doesn't look good.
            # SPACES_PER_TAB = 15  # noqa: N806 (should be lowercase)
            # tab_distance = QFontMetricsF(self.ui.edtFileContents.font()).horizontalAdvance(' ') * SPACES_PER_TAB
            # self.ui.edtFileContents.setTabStopDistance(tab_distance)
            # widget_builder.fill_edt_with_file(file_name, self.ui.edtFileContents, header='')
            self.ui.btnSave.setEnabled(False)

    def _fill_edt_with_file(self, file_name, widget):
        """Fills a plain text widget with the contents of a file.

        See widget_builder._fill_edt_with_file(). This one doesn't replace spaces with tabs.

        Args:
            file_name (str): file name
            widget (QPlainTextEdit): text edit
        """
        text = self._text_from_file(file_name)
        widget.appendPlainText(self._text_from_file(file_name))
        widget.moveCursor(QTextCursor.Start)
        self._current_original_text = text

    def _text_from_file(self, file_name):
        """Reads the file and returns the text.

        Args:
            file_name (str): file name
            widget (QPlainTextEdit): text edit
        """
        if file_name:
            with open(file_name, 'r') as file:
                return file.read()
        return ''

    def _setup_file_list(self, files):
        self.ui.lstFiles.clear()
        if files:
            for file in files:
                self.ui.lstFiles.addItem(os.path.basename(file))
            self.ui.lstFiles.setCurrentItem(self.ui.lstFiles.item(0))

    def _on_btn_delete_all(self):
        # Delete all files on disk except for the .pobs file
        folder = os.path.dirname(self.dlg_input.data.filename)
        for _, _, files in os.walk(folder):
            for f in files:
                extension = os.path.splitext(f)[1]
                if extension != '.pobs':
                    fs.removefile(os.path.join(folder, f))
        self._current_original_text = ''
        self.ui.edtFileContents.clear()
        self._files.clear()
        self._setup_file_list(self._files)
        self._update_save_button_status()
        self.ui.edtErrors.clear()

    def showEvent(self, event):  # noqa: N802
        """Restore last position and geometry when showing dialog."""
        super().showEvent(event)

    def _setup_splitter(self):
        self.ui.splitter.setSizes([200, 400])

    def accept(self):
        """Called when the OK button is clicked. Makes sure a Ugrid is selected."""
        super().accept()

    def reject(self):
        """Called when the Cancel button is clicked."""
        super().reject()


def _get_obs_coverages(dlg_input: DialogInput) -> tuple[ObsCovData, ObsCovData]:
    """Returns a tuple of dependent variable (head, concentration, or temperature) and flow coverages.

    Returns:
        (tuple[dict[list[str]]]): 2 dicts: dependent variable and flow, each with coverage and list of attribute files.
    """
    dep_var_names = []
    flow_names = []

    if dlg_input.data.model.ftype == 'GWF6':
        dep_var_names = [pest_obs_data_generator.OBS_HEAD, pest_obs_data_generator.OBS_TRANS_HEAD]
        flow_names = [pest_obs_data_generator.OBS_FLOW]
    elif dlg_input.data.model.ftype == 'GWT6':
        dep_var_names = [pest_obs_data_generator.OBS_MF6_CONC]
    elif dlg_input.data.model.ftype == 'GWE6':
        dep_var_names = [pest_obs_data_generator.OBS_MF6_TEMP]

    query = dlg_input.query  # for short
    dep_var_cov_data: ObsCovData = {}
    flow_cov_data: ObsCovData = {}
    items = tree_util.descendants_of_type(query.project_tree, xms_types=['TI_CON_MOD'])
    for item in items:
        coverage_items = tree_util.descendants_of_type(item, xms_types=['TI_COVER'])
        for coverage_item in coverage_items:
            coverage = query.item_with_uuid(coverage_item.uuid)
            att_files: dict[str, str] = query.coverage_attributes(
                cov_uuid=coverage_item.uuid, points=True, arcs=True, polygons=True, arc_groups=True
            )
            cov_path = tree_util.tree_path(coverage_item)
            for _feature_type, att_file in att_files.items():
                table_def = map_util.read_table_definition_file(att_file)
                _add_coverage_if_att_exists(coverage, cov_path, dep_var_names, table_def, att_file, dep_var_cov_data)
                _add_coverage_if_att_exists(coverage, cov_path, flow_names, table_def, att_file, flow_cov_data)
    return dep_var_cov_data, flow_cov_data


def _add_coverage_if_att_exists(
    coverage: Coverage, cov_path: str, names: list[str], table_def: dict, att_file: str, cov_data: ObsCovData
) -> None:
    """Look for attribute names in columns and, if found, add coverage, or append the attribute file to the list.

    Args:
        coverage: The coverage.
        cov_path: The coverage tree path.
        names: Attribute names.
        table_def: Table definition.
        att_file: Att file.
        cov_data: The dict we are adding to.
    """
    for column in table_def.get('columns', []):
        name = column.get('name', '')
        if name in names:
            if coverage not in cov_data:
                cov_data[coverage] = CovInfo(cov_path)
            cov_data[coverage].att_files.append(att_file)
            break
