"""Functions for creating widgets in a table."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtGui import QPalette
from PySide2.QtWidgets import (
    QCheckBox, QComboBox, QHBoxLayout, QLineEdit, QSpinBox, QTableWidget, QTableWidgetItem, QWidget
)

# 3. Aquaveo modules
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules


def add_checkbox_if_needed(row: int, table_widget: QTableWidget, checkbox_slot) -> int:
    """Adds a checkbox into the first column if the number of columns is 3.

    Args:
        row: The table row.
        table_widget: The table.
        checkbox_slot: Function to call if the checkbox state changes.

    Return:
        The column offset.
    """
    column_offset = 0
    if table_widget.columnCount() == 3:  # There's a column of checkboxes
        # We could just use a QTableWidgetItem and setCheckState but then you can actually type
        # text next to the checkbox. So we use a QCheckBox widget. But to get it to align nicely
        # in the center we have to use a layout too.
        # See https://stackoverflow.com/questions/46434577/qtablewidget-centering-cellwidgets
        check_box_widget = QWidget()
        check_box = QCheckBox('')
        layout_check_box = QHBoxLayout(check_box_widget)
        layout_check_box.addWidget(check_box)
        layout_check_box.setAlignment(Qt.AlignCenter)
        layout_check_box.setContentsMargins(0, 0, 0, 0)

        table_widget.setCellWidget(row, 0, check_box_widget)
        check_box.stateChanged.connect(checkbox_slot)
        column_offset = 1
    return column_offset


def add_combo_box_row(label, options_list, default, table_widget, locked, checkbox_slot=None):
    """Adds a row to the table containing a combo box.

    Args:
        label (str): The label string, or label next to the widget.
        options_list (list of str): Strings to put in the combo box.
        default (int): Index of the default choice to show.
        table_widget (QTableWidget): The table.
        locked (bool): True to disable the edit field.
        checkbox_slot: Function to call if the checkbox state changes.
    """
    row = table_widget.rowCount()
    table_widget.insertRow(row)

    column_offset = add_checkbox_if_needed(row, table_widget, checkbox_slot)

    widget = QTableWidgetItem(label)
    table_widget.setItem(row, column_offset, widget)

    if not locked:
        cbx = QComboBox()
        # cbx.setFrame(False) This removes the margin making it look bad
        cbx.addItems(options_list)
        cbx.setCurrentIndex(default)
        pal = cbx.palette()
        pal.setColor(QPalette.Window, pal.color(QPalette.Light))
        cbx.setPalette(pal)
        table_widget.setCellWidget(row, column_offset + 1, cbx)
    else:
        widget = QLineEdit(options_list[default])
        widget.setFrame(False)
        widget.setEnabled(False)
        table_widget.setCellWidget(row, column_offset + 1, widget)


def add_text_edit_row(label: str, default, table_widget: QTableWidget, locked: bool, checkbox_slot=None):
    """Adds a row to the table widget containing a double edit field.

    Args:
        label: The label string, or label next to the widget.
        default: The default value to show.
        table_widget: The table.
        locked: True to disable the edit field.
        checkbox_slot: Function to call if the checkbox state changes.
    """
    row = table_widget.rowCount()
    table_widget.insertRow(row)

    column_offset = add_checkbox_if_needed(row, table_widget, checkbox_slot)

    widget = QTableWidgetItem(label)
    table_widget.setItem(row, column_offset, widget)

    edt = QLineEdit(str(default))
    edt.setFrame(False)
    if locked:
        edt.setEnabled(False)
    table_widget.setCellWidget(row, column_offset + 1, edt)
    return widget


def add_double_edit_row(label, default, table_widget, locked, checkbox_slot=None):
    """Adds a row to the table widget containing a double edit field.

    Args:
        label (str): The label string, or label next to the widget.
        default (double): The default double to show.
        table_widget (QTableWidget): The table.
        locked (bool): True to disable the edit field.
        checkbox_slot: Function to call if the checkbox state changes.
    """
    row = table_widget.rowCount()
    table_widget.insertRow(row)

    column_offset = add_checkbox_if_needed(row, table_widget, checkbox_slot)

    widget = QTableWidgetItem(label)
    table_widget.setItem(row, column_offset, widget)

    edt = QLineEdit(str(default))
    edt.setFrame(False)
    validator = QxDoubleValidator(parent=edt)
    edt.setValidator(validator)
    if locked:
        edt.setEnabled(False)
    table_widget.setCellWidget(row, column_offset + 1, edt)
    return widget


def add_int_spin_row(label, default, minimum, maximum, table_widget, locked, checkbox_slot=None):
    """Adds a row to the table widget containing a int edit field.

    Args:
        label (str): The label string, or label next to the widget.
        default (int): The default int to show.
        minimum (int): The minimum limit.
        maximum (int): The maximum limit.
        table_widget (QTableWidget): The table.
        locked (bool): True to disable the edit field.
        checkbox_slot: Function to call if the checkbox state changes.
    """
    row = table_widget.rowCount()
    table_widget.insertRow(row)

    column_offset = add_checkbox_if_needed(row, table_widget, checkbox_slot)

    widget = QTableWidgetItem(label)
    table_widget.setItem(row, column_offset, widget)

    widget = QSpinBox()
    widget.setFrame(False)
    widget.setRange(minimum, maximum)
    if type(default) is str:
        widget.setSpecialValueText(default)
        widget.setValue(minimum)
    else:  # Assuming an int
        widget.setValue(default)
    if locked:
        widget.setEnabled(False)
    table_widget.setCellWidget(row, column_offset + 1, widget)
    return widget


def disable_item(row, column, table_widget):
    """Disables the item at row, column.

    Args:
        row (int): The row.
        column (int): The column
        table_widget (QTableWidget): The table.
    """
    # Disable
    flags = table_widget.item(row, column).flags()
    flags = flags & ~Qt.ItemIsEditable
    widget = table_widget.item(row, column)
    widget.setFlags(flags)
    # Set background color to grey to indicate read-only
    main_window_bg_color = QPalette().color(QPalette.Window)
    table_widget.item(row, column).setBackground(main_window_bg_color)


def disable_column(table_widget, column=0):
    """Disables all of column 0 so it is not editable.

    Args:
        table_widget (QTableWidget): The table.
        column (int): The table column.
    """
    for row in range(0, table_widget.rowCount()):
        disable_item(row, column, table_widget)
