"""CellAdder class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.mapping.cell_adder import CellAdder
from xms.mf6.misc.util import XM_NODATA


class CellAdderPolysChd(CellAdder):
    """Adds a cell BC."""
    def __init__(self, builder, cell_idx, record, period_dict: dict, area: float | None, total_overlap: float | None):
        """Initializes the class.

        Args:
            builder: The package builder.
            cell_idx: Cell index.
            record: Shapefile record for the current arc.
            period_dict: Dict of cell_idx -> list of values (a row in the list package).
            area: Area of overlap for a polygon.
            total_overlap: Total area in cell overlapped by all polygons.
        """
        super().__init__(builder)
        self.set_record(cell_idx, XM_NODATA, record, None)
        self._period_dict = period_dict
        self._area = area
        self._total_overlap = total_overlap

    def add_bc_to_cell(self) -> None:
        """Add a boundary condition to a cell."""
        column_values = self._get_column_values()
        if column_values:
            if self._area is not None:
                column_values[0] = (self._area / self._total_overlap) * column_values[0]
            self._add_chd_cell_values_to_period(column_values)

    def _add_chd_cell_values_to_period(self, values):
        """Adds a new row or adds values to the existing row (because with MF6 we can't repeat CHD cells).

        Args:
            values (list): List of values = a row in the list package.
        """
        # Add a new row or add values to the existing row because we can't repeat CHD cells
        if self._cell_idx1 not in self._period_dict:
            self._period_dict[self._cell_idx1] = values
        else:
            self._period_dict[self._cell_idx1] = [a + b for a, b in zip(self._period_dict[self._cell_idx1], values)]
