"""Logging utility functions."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.core.filesystem import filesystem as fs
from xms.testing.type_aliases import Pathlike

# 4. Local modules
from xms.mf6.misc import util

# Module variables

_log_file_handler: logging.FileHandler | None = None
"""Log file handler."""

_test_logger_name: str = ''
"""Logger name used for tests."""

_test_logger: logging.Logger | None = None
"""Logger used for tests."""


class LogContext:
    """Context manager for use with the log while testing."""
    def __init__(self, test_id: str = ''):
        """Initializer.

        Args:
            test_id: A string to uniquely identify the current test.
        """
        short_id = '.'.join(test_id.split('.')[-2:])  # class name and method name, e.g. 'TestBob.test_bob'
        self._test_id = short_id
        self.logger: logging.Logger | None = None

    def __enter__(self):
        """Context manager enter.

        Returns:
            Self.
        """
        self.logger = set_up_test_logger(self._test_id)
        return self

    def __exit__(self, exc_type, exc_value, traceback) -> None:
        """Context manager exit.

        Args:
            exc_type: Exception type.
            exc_value: Exception value.
            traceback: Exception traceback.
        """
        self.logger = None
        clear_test_logger()


def set_up_test_logger(test_id: str, folder: Pathlike = None) -> logging.Logger:
    """Set up the logger to be used during the test.

    Args:
        folder: Optional folder where test log file will be created.
        test_id: Name of the test currently running.
    """
    global _test_logger_name, _test_logger, _log_file_handler
    _test_logger_name = get_logger_name(suffix=f'-{test_id}')
    _test_logger = logging.getLogger(_test_logger_name)
    filepath = _get_log_filepath(_test_logger_name, folder)
    if filepath.is_file():
        fs.removefile(filepath)
    _log_file_handler = _create_log_file_handler(filepath)
    _test_logger.addHandler(_log_file_handler)
    return _test_logger


def clear_test_logger() -> None:
    """Set test logger to None."""
    global _test_logger_name, _test_logger, _log_file_handler
    _test_logger_name = ''
    if _test_logger and _log_file_handler:
        _log_file_handler.flush()
        _log_file_handler.close()
        _test_logger.removeHandler(_log_file_handler)
        _log_file_handler = None
        _test_logger = None


def get_logger() -> logging.Logger:
    """Return the xmsmf6 logger, properly initialized.

    Returns:
        See description.
    """
    if _test_logger:
        return _test_logger

    logger = logging.getLogger(get_logger_name())

    # Add a FileHandler to log to the xms.mf6.log file
    global _log_file_handler
    if _log_file_handler is None:
        _log_file_handler = _create_log_file_handler()
        logger.addHandler(_log_file_handler)
    return logger


def get_file_handler(testlog: logging.Logger | None = None) -> logging.FileHandler | None:
    """Return the FileHandler, or None.

    Args:
        testlog: Only used when testing.

    Returns:
        See description.
    """
    return _find_file_handler(testlog) if testlog else _log_file_handler


def get_log_file(testlog: logging.Logger | None = None) -> Pathlike:
    """Return the log filepath or None.

    Args:
        testlog: Only used when testing.

    Returns:
        See description.
    """
    file_handler = get_file_handler(testlog)
    if file_handler:
        return Path(file_handler.baseFilename)
    return None


def errors_logged(testlog: logging.Logger | None = None) -> bool:
    """Return True if the log shows there were errors.

    Args:
        testlog: Only used when testing.

    Returns:
        See description.
    """
    file_handler = get_file_handler(testlog)
    if file_handler:
        file_handler.flush()
        log_file = Path(file_handler.baseFilename)
        return log_file.is_file() and log_file.stat().st_size > 0
    return False


def get_logger_name(suffix: str = '') -> str:
    """Return the name of the logger.

    Args:
        suffix: Only used when testing. E.g. ('-test_name')
    """
    if _test_logger_name:
        return _test_logger_name
    return f'xms.mf6{suffix}'


def _get_log_filepath(logger_name: str = '', folder: Pathlike = None) -> Path:
    """Return the path to the log file, initializing _log_dir if necessary.

    Args:
        logger_name: Only used when testing.
        folder: Optional folder where test log file will be created.

    Returns:
        See description.
    """
    log_dir = folder if folder else Path(XmEnv.xms_environ_temp_directory())
    name = logger_name if logger_name else get_logger_name()
    return log_dir / f'{name}.log'


def _find_file_handler(testlog: logging.Logger) -> logging.FileHandler | None:
    """Return the first FileHandler in the logger, or None.

    Args:
        testlog: Only used when testing.

    Returns:
        See description.
    """
    file_handler = None
    for handler in testlog.handlers:
        if isinstance(handler, logging.FileHandler):
            file_handler = handler
            break
    return file_handler


def _create_log_file_handler(filepath: Pathlike = None) -> logging.FileHandler:
    """Create and return a FileHandler.

    Logging level is set at logging.ERROR so it only logs errors.

    Args:
        filepath: Only used when testing.

    Returns:
        See description.
    """
    filepath = filepath if not util.null_path(filepath) else _get_log_filepath()
    fs.removefile(filepath)  # Delete the old log file
    file_handler = logging.FileHandler(filepath)
    file_handler.setLevel(logging.ERROR)
    return file_handler
