"""Runs the PEST observation batch files."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path
from subprocess import PIPE, Popen, STDOUT

# 2. Third party modules

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as fs

# 4. Local modules
from xms.mf6.misc import log_util


def run_pest_batch_files(mfsim_dir: str, pest_obs_dir: str) -> None:
    """Runs the pest batch files.

    Args:
        mfsim_dir: Path to directory containing mfsim.nam file.
        pest_obs_dir: Path to directory containing all the PEST files.
    """
    log = log_util.get_logger()
    obs_bat_filepath = os.path.join(pest_obs_dir, 'obs.bat')
    relative_path = fs.compute_relative_path(mfsim_dir, obs_bat_filepath)
    log.info(f'Running PEST batch file "{relative_path}"')
    os.chdir(pest_obs_dir)
    command = f'"{obs_bat_filepath}"'
    process = Popen(command, cwd=pest_obs_dir, stdout=PIPE, stderr=STDOUT, shell=True)
    normal_termination = False
    lines: list[str] = []
    while True:
        line = process.stdout.readline()
        if not line:
            break
        line = line.decode('utf-8').rstrip()
        if line.find('written ok.') > -1:  # TODO: what is the last line of the last batch file if all is OK?
            normal_termination = True
        lines.append(line)

    # Write the lines to a text file
    out_filepath = Path(obs_bat_filepath).with_suffix('.out')
    with out_filepath.open('w') as out_file:
        for line in lines:
            out_file.write(f'{line}\n')

    if not normal_termination:
        log.error(f'Errors running PEST batch file. See "{str(out_filepath)}"')
    process.stdout.close()
    process.wait()


def pest_obs_dir_from_model_filename(model_filename: str) -> str:
    """Returns the path to the pest observations directory give path to model filename.

    Args:
        model_filename: File path of model file.

    Returns:
        See description
    """
    return os.path.splitext(model_filename)[0] + '_pest'
