"""Module for ExportSimThread class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['ExportSimThread']

# 1. Standard Python modules
from datetime import timedelta

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.dmi.xms_data import XmsData
from xms.gmi.data_bases.sim_base_data import SimBaseData
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.feedback_thread import ExitError, FeedbackThread
from xms.guipy.time_format import datetime_to_string
from xms.guipy.widgets.widget_builder import datetime_from_string

# 4. Local modules
from xms.ptm.components.sources_component import PtmSourcesComponent
from xms.ptm.components.traps_component import PtmTrapsComponent
from xms.ptm.file_io.control import write_control
from xms.ptm.file_io.sources.source_writer import write_sources
from xms.ptm.file_io.traps.trap_writer import write_traps
from xms.ptm.model.model import simulation_model, StopMode


class ExportSimThread(FeedbackThread):
    """Import thread."""
    def __init__(self, data: XmsData):
        """
        Construct the worker.

        Args:
            data: Interprocess communication object.
        """
        super().__init__(data)
        self.display_text |= {
            'title': 'Reading Example Control File',
            'working_prompt': 'Reading control file. Please wait...',
        }
        self._data = data
        self._ok = True
        self._stop_time: str = ''

    def _run(self):
        """Run the thread."""
        self._log.info('Exporting simulation...')

        self._check_sources()
        self._check_traps()
        if not self._ok:
            raise ExitError()

        self._write_control()
        self._write_sources()
        self._write_traps()

    def _write_control(self):
        """Write the control file."""
        self._log.info('Writing control file...')

        sim_data: SimBaseData = self._data.simulation_data
        section = simulation_model()
        section.restore_values(sim_data.global_values)

        simulation_name = self._data.simulation_name
        write_control(section, f'{simulation_name}.pcf')

        time = section.group('time')
        if time.parameter('stop_mode').value == StopMode.date_time:
            self._stop_time = time.parameter('stop_run').value
        else:
            start = datetime_from_string(time.parameter('start_run').value)
            seconds = float(time.parameter('duration').value)
            duration = timedelta(seconds=seconds)
            end = start + duration
            self._stop_time = datetime_to_string(end)

    def _check_sources(self):
        """
        Check that the source coverage can be written and log warnings if not.

        Sets self._ok=False if the sources can't be written.
        """
        coverage, data = self._data.get_linked_coverage(PtmSourcesComponent)
        if not coverage:
            self._log.error('Exporting requires a PTM Sources coverage linked to the simulation.')
            self._ok = False
            return

        have_points = data.component_id_map[TargetType.point]
        have_arcs = data.component_id_map[TargetType.arc]
        have_polygons = data.component_id_map[TargetType.polygon]

        if not have_points and not have_arcs and not have_polygons:
            self._log.error('The linked Sources coverage has no assigned features.')
            self._ok = False
            return

        sim_data: SimBaseData = self._data.simulation_data
        section = simulation_model()
        section.restore_values(sim_data.global_values)
        where = section.group('files').parameter('source_file').value
        if not where:
            self._log.error('No source file name specified in the model control.')
            self._ok = False
            return

    def _write_sources(self):
        """Write the sources."""
        sim_data: SimBaseData = self._data.simulation_data
        section = simulation_model()
        section.restore_values(sim_data.global_values)
        where = section.group('files').parameter('source_file').value

        coverage, data = self._data.get_linked_coverage(PtmSourcesComponent)
        write_sources(coverage, data, self._stop_time, where)

    def _check_traps(self):
        """
        Check that the trap coverage is okay and log warnings if not.

        Sets self._ok=False if an unrecoverable error is encountered.
        """
        coverage, data = self._data.get_linked_coverage(PtmTrapsComponent)
        if not coverage:
            return

        sim_data: SimBaseData = self._data.simulation_data
        section = simulation_model()
        section.restore_values(sim_data.global_values)
        where = section.group('files').parameter('trap_file').value
        if not where:
            self._log.error('No trap file name specified in the model control.')
            self._ok = False
            return

    def _write_traps(self):
        """Write the traps if necessary."""
        coverage, data = self._data.get_linked_coverage(PtmTrapsComponent)
        if not coverage:
            return

        sim_data: SimBaseData = self._data.simulation_data
        section = simulation_model()
        section.restore_values(sim_data.global_values)
        where = section.group('files').parameter('trap_file').value

        coverage, data = self._data.get_linked_coverage(PtmTrapsComponent)
        write_traps(coverage, data, where)
