"""Module for running the low-level reader and converting its format to high-level sources."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['read_sources']

# 1. Standard Python modules
from itertools import groupby
from pathlib import Path
from typing import cast, Iterator, TextIO

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.component_builders.coverage_component_builder import CoverageComponentBuilder
from xms.data_objects.parameters import Coverage
from xms.gmi.data_bases.coverage_base_data import CoverageBaseData
from xms.guipy.data.target_type import TargetType
from xms.guipy.time_format import datetime_to_string
from xms.ptmio.source.source_reader import read_sources as pio_read
from xms.ptmio.source.sources import (
    InstantMassSource, LineMassDatum, LineMassSource, PointMassSource, PolygonMassSource, Pt3d
)

# 4. Local modules
from xms.ptm.components.sources_component import PtmSourcesComponent
from xms.ptm.model.model import source_model


def read_sources(where: str | Path | TextIO) -> tuple[Coverage, PtmSourcesComponent]:
    """
    Read the sources out of a file.

    Args:
        where: Where to read from.

    Returns:
        The sources that were read.
    """
    pio_sources = pio_read(where)
    builder = CoverageComponentBuilder(PtmSourcesComponent, 'PTM Sources')

    _process_instants(pio_sources.instant_sources, builder)
    _process_points(pio_sources.point_sources, builder)
    _process_vertical_lines(pio_sources.line_sources, builder)
    _process_horizontal_lines(pio_sources.line_sources, builder)
    _process_polygons(pio_sources.polygon_sources, builder)

    coverage, component = builder.build(using_xms_data=True)
    return coverage, component


def _process_instants(pio_instants: list[InstantMassSource], builder: CoverageComponentBuilder):
    """Process the instant sources."""
    for pio_instant in pio_instants:
        _process_instant(pio_instant, builder)


def _process_instant(pio_instant: InstantMassSource, builder: CoverageComponentBuilder):
    """Process an instant source."""
    location = pio_instant.instructions[0].location
    attributes = source_model().point_parameters
    attributes.group('instant').is_active = True
    attributes.group('instant').parameter('id').value = pio_instant.source_id
    attributes.group('instant').parameter('label').value = pio_instant.label

    table = []
    for instruction in pio_instant.instructions:
        row = [
            datetime_to_string(instruction.time), *instruction.location, instruction.parcel_mass, instruction.h_radius,
            instruction.v_radius, instruction.mass_rate, instruction.grain_size, instruction.stdev, instruction.density,
            instruction.velocity, instruction.initiation, instruction.deposition
        ]
        table.append(row)

    attributes.group('instant').parameter('instructions').value = table

    data = cast(CoverageBaseData, builder.data)
    component_id = data.add_feature(TargetType.point, attributes.extract_values(), 'instant')
    builder.add_point(*location, component_id=component_id)


def _process_points(pio_points: list[PointMassSource], builder: CoverageComponentBuilder):
    """Process the point sources."""
    for pio_point in pio_points:
        _process_point(pio_point, builder)


def _process_point(pio_point: PointMassSource, builder: CoverageComponentBuilder):
    """Process a point source."""
    location = pio_point.instructions[0].location
    attributes = source_model().point_parameters
    attributes.group('point').is_active = True
    attributes.group('point').parameter('id').value = pio_point.source_id
    attributes.group('point').parameter('label').value = pio_point.label

    table = []
    for instruction in pio_point.instructions:
        row = [
            datetime_to_string(instruction.time), *instruction.location, instruction.parcel_mass, instruction.h_radius,
            instruction.v_radius, instruction.mass_rate, instruction.grain_size, instruction.stdev, instruction.density,
            instruction.velocity, instruction.initiation, instruction.deposition
        ]
        table.append(row)

    attributes.group('point').parameter('instructions').value = table

    data = cast(CoverageBaseData, builder.data)
    component_id = data.add_feature(TargetType.point, attributes.extract_values(), 'point')
    builder.add_point(*location, component_id=component_id)


def _process_vertical_lines(pio_lines: list[LineMassSource], builder: CoverageComponentBuilder):
    """Process the vertical line sources."""
    pio_lines = [line for line in pio_lines if line.datum != LineMassDatum.none]

    # XMS doesn't support vertical arcs natively, so the PTM interface fakes it by letting you assign a pair of
    # elevations to a point. But you can have multiple segments on a horizontal arc, so we want to support multiple
    # segments on a vertical arc too. Faking *that* entails stashing all the attributes for each segment at a given
    # location onto a single point, which means we have to deal with a potentially many-to-one mapping between lines in
    # the file and points in the coverage.

    def key(line: LineMassSource):
        start = line.instructions[0].start
        return start[0], start[1]

    pio_lines.sort(key=key)
    groups = groupby(pio_lines, key=key)
    for _location, pio_line in groups:
        _process_vertical_line(pio_line, builder)


def _process_vertical_line(lines: Iterator[LineMassSource], builder: CoverageComponentBuilder):
    """Process a vertical line source."""
    datum_names = {
        LineMassDatum.bed_datum: 'Bed Datum',
        LineMassDatum.depth_distributed: 'Depth Distributed',
        LineMassDatum.surface_datum: 'Surface Datum',
    }
    table = []
    for line in lines:
        for instruction in line.instructions:
            # If the indexes of these change, the sorting and grouping operations below might need to change too.
            row = [
                line.label, line.source_id, datum_names[line.datum],
                datetime_to_string(instruction.time), instruction.start[0], instruction.start[1], instruction.start[2],
                instruction.end[2], instruction.parcel_mass, instruction.h_radius, instruction.mass_rate,
                instruction.grain_size, instruction.stdev, instruction.density, instruction.velocity,
                instruction.initiation, instruction.deposition
            ]
            table.append(row)

    table.sort(key=lambda r: (r[0], r[3]))
    location = tuple(table[0][4:7])
    location = cast(Pt3d, location)
    attributes = source_model().point_parameters
    attributes.group('vertical_line').is_active = True
    attributes.group('vertical_line').parameter('instructions').value = table

    data = cast(CoverageBaseData, builder.data)
    component_id = data.add_feature(TargetType.point, attributes.extract_values(), 'vertical_line')
    builder.add_point(*location, component_id=component_id)


def _process_horizontal_lines(pio_horizontal_lines: list[LineMassSource], builder: CoverageComponentBuilder):
    """Process the horizontal line sources."""
    pio_horizontal_lines = [line for line in pio_horizontal_lines if line.datum == LineMassDatum.none]
    for pio_horizontal_line in pio_horizontal_lines:
        _process_horizontal_line(pio_horizontal_line, builder)


def _process_horizontal_line(pio_horizontal_line: LineMassSource, builder: CoverageComponentBuilder):
    """Process a horizontal line source."""
    start = pio_horizontal_line.instructions[0].start
    end = pio_horizontal_line.instructions[0].end
    attributes = source_model().arc_parameters
    attributes.group('horizontal_line').is_active = True
    attributes.group('horizontal_line').parameter('id').value = pio_horizontal_line.source_id
    attributes.group('horizontal_line').parameter('label').value = pio_horizontal_line.label

    table = []
    for instruction in pio_horizontal_line.instructions:
        row = [
            datetime_to_string(instruction.time), instruction.start[0], instruction.start[1], instruction.end[0],
            instruction.end[1], instruction.start[2], instruction.parcel_mass, instruction.v_radius,
            instruction.mass_rate, instruction.grain_size, instruction.stdev, instruction.density, instruction.velocity,
            instruction.initiation, instruction.deposition
        ]
        table.append(row)

    attributes.group('horizontal_line').parameter('instructions').value = table

    data = cast(CoverageBaseData, builder.data)
    component_id = data.add_feature(TargetType.arc, attributes.extract_values(), 'horizontal_line')
    builder.add_point_string([start, end], component_id=component_id)


def _process_polygons(pio_polygons: list[PolygonMassSource], builder: CoverageComponentBuilder):
    """Process the polygon sources."""
    for pio_polygon in pio_polygons:
        _process_polygon(pio_polygon, builder)


def _process_polygon(pio_polygon: PolygonMassSource, builder: CoverageComponentBuilder):
    """Process a polygon source."""
    locations = pio_polygon.instructions[0].points
    attributes = source_model().polygon_parameters
    attributes.group('polygon').is_active = True
    attributes.group('polygon').parameter('id').value = pio_polygon.source_id
    attributes.group('polygon').parameter('label').value = pio_polygon.label

    table = []
    for instruction in pio_polygon.instructions:
        row = [
            datetime_to_string(instruction.time),
            instruction.points[0][2],  # Z component of first point
            instruction.parcel_mass,
            instruction.v_radius,
            instruction.mass_rate,
            instruction.grain_size,
            instruction.stdev,
            instruction.density,
            instruction.velocity,
            instruction.initiation,
            instruction.deposition
        ]
        table.append(row)

    attributes.group('polygon').parameter('instructions').value = table

    data = cast(CoverageBaseData, builder.data)
    component_id = data.add_feature(TargetType.polygon, attributes.extract_values(), 'polygon')
    builder.add_polygon([locations], component_id=component_id)
