"""Impoundment data."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel
from xms.tool_core.table_definition import FloatColumnType, StringColumnType, TableDefinition

# 4. Local modules
from xms.rsm.data import bc_val_data_def as bvdd
from xms.rsm.data import monitor_data_def as mdd
from xms.rsm.data.hpm_data_def import add_hpm_to_section
from xms.rsm.data.mesh_data_def import add_rain_et_to_group


class _BasinDataDef:
    """A class to define the impoundment data for the generic model."""
    def __init__(self):
        self.generic_model = GenericModel(exclusive_polygon_conditions=False)
        self._pp = self.generic_model.polygon_parameters
        self._poly_data_def()

    def _poly_data_def(self):
        """Defines the polygon data for the generic model."""
        self._basin()
        rain_gp = self._pp.add_group(group_name='rain', label='Basin rain')
        et_gp = self._pp.add_group(group_name='refet', label='Basin evapotranspiration')
        add_rain_et_to_group(rain_gp, et_gp)
        add_hpm_to_section(self._pp, group_label='Basin HPM (Hydraulic Process Module)', include_index=False)
        self._boundary_conditions()
        self._monitor()

    def _basin(self):
        pp = self._pp
        grp = pp.add_group(group_name='basin', label='Basin')
        grp.add_text(name='label', label='Label (<label>)')
        grp.add_float(name='init_head', label='Initial water level (ft) (<intial_head>)', default=0.0)
        grp.add_float(name='area', label='Area (ft^2) (<area>)', default=0.0)
        label = 'Average land surface elevation (ft) (<elev>)'
        grp.add_float(name='elev', label=label, default=0.0)
        label = 'Height above average to highest peak (ft) (<above>)'
        grp.add_float(name='above', label=label, default=0.0)
        label = 'Depth below average to lowest depression (ft) (<below>)'
        grp.add_float(name='below', label=label, default=0.0)
        grp.add_float(name='sscoef', label='Soil storage coefficient (<sscoef>)', default=0.2)
        # inputs for SSTable
        cols = [FloatColumnType(header='Stage', default=0.0), FloatColumnType(header='Volume', default=0.0)]
        td = TableDefinition(cols)
        default_vals = []
        grp.add_table(name='sv', label='Stage-volume table (<sv>)', table_definition=td, default=default_vals)

    def _boundary_conditions(self):
        bc_gp = self._pp.add_group(group_name='bc', label='Basin Boundary Conditions')
        bc_gp.add_text(name='label', label='Label (<label>)')
        bc_gp.add_integer(name='bc_id', label='Boundary condition id (<id>)', default=-1)
        bc_gp.add_text(name='tag', label='Tag (<tag>)', default='')
        opts = ['basinHeadBC', 'basinsource']
        opt_par = bc_gp.add_option(name='bc_type', label='BC type', default=opts[0], options=opts)
        pckg_opts = ['Not specified', 'areaCorrected']
        pckg_par = bc_gp.add_option(name='package', label='Source package', default=pckg_opts[0], options=pckg_opts)
        flags = {opt: False for opt in opts}
        flags['basinsource'] = True
        pckg_par.add_dependency(opt_par, flags)
        # table of lake ids for areaCorrected
        pckg_flags = {o: False for o in pckg_opts}
        pckg_flags['areaCorrected'] = True
        p_spec_lakes = bc_gp.add_boolean(name='specify_lakes', label='Specify lakes', default=False)
        p_spec_lakes.add_dependency(pckg_par, pckg_flags)
        td = TableDefinition([StringColumnType(header='Lake label', default='')])
        p = bc_gp.add_table(name='lakes', label='Lakes', default=[], table_definition=td)
        td_flags = {True: True, False: False}
        p.add_dependency(p_spec_lakes, td_flags)

        bvdd.add_bc_val_to_group(group=bc_gp)

    def _monitor(self):
        pp = self._pp
        outputs = [
            ('basinsto', 'Monitor storage volume (<basinsto>)'),
            ('initbasinsto', 'Monitor initial storage volume (<initbasinsto>)'),
            ('head', 'Monitor head (<head>)'),
            ('recharge', 'Monitor recharge volume (<recharge>)'),
            ('runoff', 'Monitor runoff volume (<runoff>)'),
            ('wsupply', 'Monitor water supply withdrawals (<wsupply>)'),
            ('basinrecharge', 'Monitor recharge volume (<basinrecharge>)'),
            ('basinrunoff', 'Monitor runoff received (<basinrunoff>)'),
            ('basinwsupply', 'Monitor water supply withdrawals (<basinwsupply>)'),
            ('basinrainvol', 'Monitor rain volume (<basinrainvol>)'),
            ('basinetvol', 'Monitor ET volume (<basinetvol>)'),
        ]
        for name, label in outputs:
            gp = pp.add_group(group_name=name, label=label)
            mdd.add_output_to_group(gp)


def generic_model():
    """Gets a generic model for the boundary conditions coverage.

    Returns:
        (xms.gmi.data.generic_model.GenericModel): the generic model class
    """
    dd = _BasinDataDef()
    return dd.generic_model
