"""Data class for RSM hpm natural system property."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass, field

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


@dataclass
class HpmNaturalSystemData:
    """Holds data for RSM hpm natural system property."""
    @dataclass
    class Layer1Nsm:
        """Layer 1 natural system model parameters."""
        kw: float = 0.0
        rd: float = 0.0
        xd: float = 0.0
        pd: float = 0.0
        kveg: float = 0.0
        fld_cap: float = 0.0
        imax: float = 0.0

    @dataclass
    class Unsat:
        """Unsat natural system model parameters."""
        ew: float = 0.0
        kw: float = 0.0
        rd: float = 0.0
        xthresh: float = 0.0
        pthresh: float = 0.0
        pd: float = 0.0
        wilt: float = 0.0
        kveg: float = 0.0

    @dataclass
    class Nam:
        """Nam natural system model parameters."""
        etcoef: float = 0.0
        k0inf: float = 0.0
        imax: float = 0.0
        umax: float = 0.0
        lmax: float = 0.0
        tof: float = 0.0
        tif: float = 0.0
        tg: float = 0.0
        cqof: float = 0.0
        ckol: float = 0.0
        ckif: float = 0.0
        ckbf: float = 0.0

    @dataclass
    class Layer5:
        """Layer 5 natural system model parameters."""
        ew: float = 0.0
        kw: float = 0.0
        rd: float = 0.0
        xd: float = 0.0
        pd: float = 0.0
        kveg: float = 0.0

    type: str = ''
    layer1: Layer1Nsm = field(default_factory=Layer1Nsm)
    unsat: Unsat = field(default_factory=Unsat)
    nam: Nam = field(default_factory=Nam)
    layer5: Layer5 = field(default_factory=Layer5)


@dataclass
class HpmAgriculturalSystemData:
    """Holds data for RSM hpm agricultural system property."""
    @dataclass
    class Afsirs:
        """afsirs system model parameters."""
        subday: bool = False
        coupled: bool = False
        # crop
        crop_label: str = ''
        crop_id: int = -1
        j1: str = ''
        jn: str = ''
        crop_type: str = ''
        depth1: float = 0.0
        depth2: float = 0.0
        kctbl: list = field(default_factory=list)
        awtbl: list = field(default_factory=list)
        # irrigation
        afirr_label: str = ''
        wtd: float = 0.0
        netgross: bool = False
        irrmeth_id: int = -1
        eff: float = 0.0
        arzi: float = 0.0
        exir: float = 0.0
        drinc: float = 0.0
        crown: float = 0.0
        irrmgmt_label: str = ''
        trig_code: int = 0
        value_fixed: float = 0.0
        value_asw: float = 0.0
        # soil
        soil_label: str = ''
        depth: float = 0.0
        minwc: float = 0.0
        maxwc: float = 0.0
        cond: int = 0

    @dataclass
    class PumpedDitch:
        """pumpedditch system model parameters."""
        percent_area: float = 0.0
        rks: float = 0.0
        fc_pump: float = 0.0
        ws_pump: float = 0.0
        bottom: float = 0.0
        max_level: float = 0.0
        min_level: float = 0.0
        fc_pump_on: float = 0.0
        fc_pump_off: float = 0.0
        ws_pump_on: float = 0.0
        ws_pump_off: float = 0.0
        runoff: str = ''
        bi_dir_seep: bool = False

    @dataclass
    class AgImp:
        """agimp system model parameters."""
        field_area: float = 0.0
        rks: float = 0.0
        height: float = 0.0
        r25y3d: float = 0.0
        allow: float = 0.0
        s: float = 0.0
        runoff: str = ''

    @dataclass
    class EaaRunoffDemand:
        """eaaRunoffDemand system model parameters."""
        kmax: float = 0.0
        owpond: float = 0.0
        drz: float = 0.0
        srz: float = 0.0
        kcoef_id: str = ''
        kcalib_id: str = ''
        label: str = ''

    ag_type: str = ''
    afsirs: Afsirs = field(default_factory=Afsirs)
    pumped_ditch: PumpedDitch = field(default_factory=PumpedDitch)
    agimp: AgImp = field(default_factory=AgImp)
    eaa_runoff_demand: EaaRunoffDemand = field(default_factory=EaaRunoffDemand)


@dataclass
class HpmUrbanSystemData:
    """Holds data for RSM hpm urban system property."""
    @dataclass
    class Imperv:
        """Impervious area HPM parameters."""
        dirconn: int = 1  # 0=DCIA, 1=UCIA
        isto: float = 0.01
        isto1: float = 0.01
        sdet: float = 0.1
        sdet1: float = 0.1

    @dataclass
    class Mbrcell:
        """Multi-basin routing HPM parameters."""
        route: str = ''
        tc: int = 3600
        kveg: float = 0.0
        d_shal: float = 0.5
        d_deep: float = 2.0
        fld_cap: float = 0.2

    @dataclass
    class Cu:
        """Consumptive use HPM parameters."""
        label: str = ''
        percentarea: float = 0.0
        wsupply: str = ''
        sewer: str = ''
        trigcat: str = ''
        const_value: float = 0.0
        fracloss: float = 0.0
        septic: bool = False

    @dataclass
    class Urbandet:
        """Urban detention HPM parameters."""
        percentarea: float = 0.0
        runoff: str = ''
        rks: float = 0.001
        wlen: float = 0.0
        angle: float = 0.0
        top: float = 0.0
        apx: float = 0.0

    imperv: Imperv = field(default_factory=Imperv)
    mbrcell: Mbrcell = field(default_factory=Mbrcell)
    cu: Cu = field(default_factory=Cu)
    urbandet: Urbandet = field(default_factory=Urbandet)


@dataclass
class HpmData:
    """Holds data for RSM hpm property."""
    hpm_type: str = ''
    ns: HpmNaturalSystemData = field(default_factory=HpmNaturalSystemData)
    ag: HpmAgriculturalSystemData = field(default_factory=HpmAgriculturalSystemData)
    urban: HpmUrbanSystemData = field(default_factory=HpmUrbanSystemData)
