"""HPM data definitions methods."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import Group, MAX_FLOAT, Section
from xms.guipy.widgets.table_with_tool_bar import FloatColumnType, TableDefinition

# 4. Local modules
from xms.rsm.data.hpm_data import HpmData, HpmNaturalSystemData

HPM_TYPE_NONE = 'Not specified'
HPM_TYPE_NS = 'Natural System'
HPM_TYPE_AG = 'Agricultural'
HPM_TYPE_URBAN = 'Urban'
HPM_TYPES = [HPM_TYPE_NONE, HPM_TYPE_NS, HPM_TYPE_AG, HPM_TYPE_URBAN]

NS_TYPE_LAY1 = 'layer1nsm'
NS_TYPE_UNSAT = 'unsat'
NS_TYPE_NAM = 'nam'
NS_TYPE_LAY5 = 'layer5'
NS_TYPE_LAYPC = 'layerpc'
NS_TYPES = [NS_TYPE_LAY1, NS_TYPE_UNSAT, NS_TYPE_NAM, NS_TYPE_LAY5, NS_TYPE_LAYPC]

AG_TYPE_AFS = 'afsirs'
AG_TYPE_RAM = 'ramcc'
AG_TYPE_PD = 'pumped ditch'
AG_TYPE_IMP = 'agimp'
AG_TYPE_EAA = 'eaaRunoffDemand'
AG_TYPES = [AG_TYPE_AFS, AG_TYPE_RAM, AG_TYPE_PD, AG_TYPE_IMP, AG_TYPE_EAA]

CROP_PER = 'perennial'
CROP_ANN = 'annual'
CROP_TYPES = [CROP_PER, CROP_ANN]

IRR_OPT = '0 - optimum irrigation'
IRR_FIX_RATE = '1 - fixed irrigation rate'
IRR_FIX_DEF = '2 - fixed soil water deficit'
IRR_TYPES = [IRR_OPT, IRR_FIX_RATE, IRR_FIX_DEF]

SOIL_COND_AVE = '1 - average'
SOIL_COND_MIN = '2 - minwc'
SOIL_COND_MAX = '3 - maxwc'
SOIL_COND_OPTS = [SOIL_COND_AVE, SOIL_COND_MIN, SOIL_COND_MAX]

URBAN_TYPE_IMPERV = 'imperv'
URBAN_TYPE_MBRCELL = 'mbrcell'
URBAN_TYPE_CU = 'cu'
URBAN_TYPE_URBANDET = 'urbandet'
URBAN_TYPES = [URBAN_TYPE_IMPERV, URBAN_TYPE_MBRCELL, URBAN_TYPE_CU, URBAN_TYPE_URBANDET]

DIRCON_0 = '0 - DCIA (directly connected)'
DIRCON_1 = '1 - UCIA (undirectly connected)'
DIRCON_TYPES = [DIRCON_0, DIRCON_1]

CU_HI = 'HI'
CU_LI = 'LI'
CU_TYPES = [CU_HI, CU_LI]

CU_TRIG_NA = 'Not specified'
CU_TRIG_CU = 'cu'
CU_TRIG_AG = 'ag'
CU_TRIG_TYPES = [CU_TRIG_NA, CU_TRIG_CU, CU_TRIG_AG]


class _HpmNaturalSystemDataDef:
    def __init__(self):
        self._load_prop = {
            NS_TYPE_LAY1: self._load_layer1nsm,
            NS_TYPE_UNSAT: self._load_unsat,
            NS_TYPE_NAM: self._load_nam,
            NS_TYPE_LAY5: self._load_layer5,
            NS_TYPE_LAYPC: self._load_layerpc
        }

    def _make_flags(self, the_type):
        flags = {t: False for t in NS_TYPES}
        flags[the_type] = True
        return flags

    def natural_system_props(self, gp, hpm_type_par):
        hpm_type_flags = {t: False for t in HPM_TYPES}
        hpm_type_flags[HPM_TYPE_NS] = True
        opt_par = gp.add_option(name='ns_type', label='Natural system type', default=NS_TYPES[0], options=NS_TYPES)
        opt_par.add_dependency(parent=hpm_type_par, flags=hpm_type_flags)
        # layer1nsm
        flags = self._make_flags(NS_TYPE_LAY1)
        names, labels, highs = self._layer1nsm_props()
        _add_props(names, labels, highs, gp, opt_par, flags)
        # unsat
        flags = self._make_flags(NS_TYPE_UNSAT)
        names, labels, highs = self._unsat_props()
        _add_props(names, labels, highs, gp, opt_par, flags)
        # nam
        flags = self._make_flags(NS_TYPE_NAM)
        names, labels, highs = self._nam_props()
        _add_props(names, labels, highs, gp, opt_par, flags)
        # layer5
        flags = self._make_flags(NS_TYPE_LAY5)
        names, labels, highs = self._layer5_props()
        _add_props(names, labels, highs, gp, opt_par, flags)
        # layerpc has no properties

    def _layer1nsm_props(self):
        names = ['kw', 'rd', 'xd', 'pd', 'kveg', 'fld_cap', 'imax']
        labels = [
            'Maximum crop coefficient (<kw>)', 'Shallow root zone depth, ft (<rd>)', 'Extinction depth, ft (<xd>)',
            'Open water ponding depth, ft (<pd>)', 'Vegetation crop coefficient (<kveg>)',
            'Water content field capacity (<fld_cap>)', 'Maximum interception (<imax>)'
        ]
        highs = [1.0, 7.0, 10.0, 2.0, 1.0, 0.5, 1.0]
        return names, labels, highs

    def _unsat_props(self):
        names = ['ew', 'kw', 'rd', 'xthresh', 'pthresh', 'pd', 'wilt', 'kveg']
        names = [f'unsat_{n}' for n in names]
        labels = [
            'Extractable water (<ew>)', 'Maximum crop coefficient for open water (<kw>)',
            'Shallow root zone depth, ft (<rd>)', 'Soil water content when ET ceases (<xthresh>)',
            'Soil water content when Kc begins to descrease from kveg to 0 (<pthresh>)',
            'Open water ponding depth, ft (<pd>)', 'Soil water content at wilting point (<wilt>)',
            'Vegetation crop coefficient (<kveg>)'
        ]
        highs = [1.0, 1.0, 2.0, 0.3, 0.5, 2.0, 0.1, 1.0]
        return names, labels, highs

    def _nam_props(self):
        names = ['etcoef', 'k0inf', 'imax', 'umax', 'lmax', 'tof', 'tif', 'tg', 'cqof', 'ckol', 'ckif', 'ckbf']
        names = [f'nam_{n}' for n in names]
        labels = [
            'ET crop coefficient (<etcoef>)', 'Maximum infiltration rate (<k0inf>)',
            'Maximum interception storage (<imax>)', 'Maximum upper zone moisture content (<umax>)',
            'Maximum lower zone moisture content (<lmax>)',
            'Threshold value of lower zone storage L for overland flow (<tof>)',
            'Threshold value of lower zone storage L for interflow (<tif>)',
            'Threshold value of lower zone storage L for groundwater recharge (<tg>)',
            'Overland flow runoff coefficient (<cqof>)', 'Time constant for overland flow routing, hrs (<ckol>)',
            'Time constant for interflow, hrs (<ckif>)', 'Time constant for groundwater routing, hrs (<ckbf>)'
        ]
        highs = [1.0, 1.0, 1.0, 1.0, 20.0, 1.0, 1.0, 1.0, 1.0, 2000.0, 2000.0, 5000.0]
        return names, labels, highs

    def _layer5_props(self):
        names = ['ew', 'kw', 'rd', 'xd', 'pd', 'kveg']
        names = [f'layer5_{n}' for n in names]
        labels = [
            'Extractable water (<ew>)', 'Maximum crop coefficient for open water (<kw>)',
            'Shallow root zone depth, ft (<rd>)', 'Extinction depth below which no ET occurs, ft (<xd>)',
            'Open water ponding depth, ft (<pd>)', 'Vegetation crop coefficient (<kveg>)'
        ]
        highs = [1.0, 1.0, 2.0, 10.0, 2.0, 1.0]
        return names, labels, highs

    def load_ns_data(self, group: Group, hpm_data: HpmData):
        ns = hpm_data.ns
        ns.type = group.parameter('ns_type').value
        self._load_prop[ns.type](group, ns)

    def _load_layer1nsm(self, group: Group, ns: HpmNaturalSystemData):
        ns.layer1.kw = group.parameter('kw').value
        ns.layer1.rd = group.parameter('rd').value
        ns.layer1.xd = group.parameter('xd').value
        ns.layer1.pd = group.parameter('pd').value
        ns.layer1.kveg = group.parameter('kveg').value
        ns.layer1.fld_cap = group.parameter('fld_cap').value
        ns.layer1.imax = group.parameter('imax').value

    def _load_unsat(self, group: Group, ns: HpmNaturalSystemData):
        ns.unsat.ew = group.parameter('unsat_ew').value
        ns.unsat.kw = group.parameter('unsat_kw').value
        ns.unsat.rd = group.parameter('unsat_rd').value
        ns.unsat.xthresh = group.parameter('unsat_xthresh').value
        ns.unsat.pthresh = group.parameter('unsat_pthresh').value
        ns.unsat.pd = group.parameter('unsat_pd').value
        ns.unsat.wilt = group.parameter('unsat_wilt').value
        ns.unsat.kveg = group.parameter('unsat_kveg').value

    def _load_nam(self, group: Group, ns: HpmNaturalSystemData):
        ns.nam.etcoef = group.parameter('nam_etcoef').value
        ns.nam.k0inf = group.parameter('nam_k0inf').value
        ns.nam.imax = group.parameter('nam_imax').value
        ns.nam.umax = group.parameter('nam_umax').value
        ns.nam.lmax = group.parameter('nam_lmax').value
        ns.nam.tof = group.parameter('nam_tof').value
        ns.nam.tif = group.parameter('nam_tif').value
        ns.nam.tg = group.parameter('nam_tg').value
        ns.nam.cqof = group.parameter('nam_cqof').value
        ns.nam.ckol = group.parameter('nam_ckol').value
        ns.nam.ckif = group.parameter('nam_ckif').value
        ns.nam.ckbf = group.parameter('nam_ckbf').value

    def _load_layer5(self, group: Group, ns: HpmNaturalSystemData):
        ns.layer5.ew = group.parameter('layer5_ew').value
        ns.layer5.kw = group.parameter('layer5_kw').value
        ns.layer5.rd = group.parameter('layer5_rd').value
        ns.layer5.xd = group.parameter('layer5_xd').value
        ns.layer5.pd = group.parameter('layer5_pd').value
        ns.layer5.kveg = group.parameter('layer5_kveg').value

    def _load_layerpc(self, group: Group, ns: HpmNaturalSystemData):
        pass


class _HpmAgriculturalDataDef:
    def __init__(self):
        self._load_prop = {
            AG_TYPE_AFS: self._load_afsirs_data,
            AG_TYPE_RAM: self._load_afsirs_data,
            AG_TYPE_PD: self._load_pumped_ditch_data,
            AG_TYPE_IMP: self._load_agimp_data,
            AG_TYPE_EAA: self._load_eaa_runoff_demand_data
        }

    def agricultural_props(self, gp, hpm_type_par):
        hpm_type_flags = {t: False for t in HPM_TYPES}
        hpm_type_flags[HPM_TYPE_AG] = True
        opt_par = gp.add_option(name='ag_type', label='Agricultural type', default=AG_TYPES[0], options=AG_TYPES)
        opt_par.add_dependency(parent=hpm_type_par, flags=hpm_type_flags)
        self._afsirs_props(gp, opt_par)
        self._pumped_ditch_props(gp, opt_par)
        self._agimp_props(gp, opt_par)
        self._eaa_runoff_demand_props(gp, opt_par)

    def _afsirs_props(self, gp, ag_opt_par):
        parameters = []
        par = parameters.append
        # dependency flags for the ag type
        flags = {t: False for t in AG_TYPES}
        flags[AG_TYPE_AFS] = True
        flags[AG_TYPE_RAM] = True

        par(gp.add_boolean(name='subday', label='Model is running subday timestep (<subday>)', default=False))
        lab = 'Is water table coupled with water table in mesh cell (<coupled>)'
        par(gp.add_boolean(name='coupled', label=lab, default=False))
        # add the other properties
        self._afsirs_crop_props(gp, par)
        self._afsirs_irrigation_props(gp, par)
        self._afsirs_soil_props(gp, par, ag_opt_par, flags)

        # add dependency to the ag type
        for p in parameters:
            p.add_dependency(parent=ag_opt_par, flags=flags)

    def _afsirs_crop_props(self, gp, par):
        # crop parameters
        par(gp.add_text(name='crop_label', label='Crop label (<label>'))
        par(gp.add_integer(name='crop_id', label='Crop id <id>', default=-1))
        lab = 'Crop season start date (MM-DD) [example 03-07] (<j1>)'
        par(gp.add_text(name='j1', label=lab, default='01-01'))
        lab = 'Crop season end date (MM-DD) [example 10-12] (<jn>)'
        par(gp.add_text(name='jn', label=lab, default='12-31'))
        lab = 'Crop type (<crop_type>)'
        crop_opt = gp.add_option(name='crop_type', label=lab, default=CROP_TYPES[0], options=CROP_TYPES)
        par(crop_opt)
        flags = {CROP_PER: True, CROP_ANN: False}
        # perennial crop parameters
        p = gp.add_float(
            name='depth1_per', label='Irrigated soil depth, in (<depth1>)', default=0.0, low=0.0, high=48.0
        )
        p.add_dependency(parent=crop_opt, flags=flags)
        p = gp.add_float(name='depth2_per', label='Root depth, in (<depth2>)', default=0.0, low=0.0, high=48.0)
        p.add_dependency(parent=crop_opt, flags=flags)
        lab = 'ET reference crop correction coefficients (<kctbl>)'
        cols = [FloatColumnType(header='Coefficients', default=0.0, tool_tip=lab)]
        table_def = TableDefinition(cols, fixed_row_count=12)
        def_vals = [[0.0] for _ in range(12)]
        p = gp.add_table(name='kctbl_per', label=lab, default=def_vals, table_definition=table_def)
        p.add_dependency(parent=crop_opt, flags=flags)
        lab = 'Allowable soil water depletion before irrigation (<awtbl>)'
        cols = [FloatColumnType(header='Depletion', default=0.0, tool_tip=lab)]
        table_def = TableDefinition(cols, fixed_row_count=12)
        p = gp.add_table(name='awtbl_per', label=lab, default=def_vals, table_definition=table_def)
        p.add_dependency(parent=crop_opt, flags=flags)
        # annual crop parameters
        flags = {CROP_PER: False, CROP_ANN: True}
        lab = 'Early season irrigated soil depth, in (<depth1>)'
        p = gp.add_float(name='depth1_ann', label=lab, default=0.0, low=0.0, high=48.0)
        p.add_dependency(parent=crop_opt, flags=flags)
        lab = 'Late season irrigated soil depth, in (<depth2>)'
        p = gp.add_float(name='depth2_ann', label=lab, default=0.0, low=0.0, high=48.0)
        p.add_dependency(parent=crop_opt, flags=flags)
        lab = 'Peak crop ET (<kctbl>)'
        cols = [FloatColumnType(header='Peak ET', default=0.0, tool_tip=lab)]
        table_def = TableDefinition(cols, fixed_row_count=6)
        def_vals = [[0.0] for _ in range(6)]
        p = gp.add_table(name='kctbl_ann', label=lab, default=def_vals, table_definition=table_def)
        p.add_dependency(parent=crop_opt, flags=flags)
        lab = 'Allowable soil water depletion by stage (<awtbl>)'
        cols = [FloatColumnType(header='Depletion', default=0.0, tool_tip=lab)]
        table_def = TableDefinition(cols, fixed_row_count=6)
        p = gp.add_table(name='awtbl_ann', label=lab, default=def_vals, table_definition=table_def)
        p.add_dependency(parent=crop_opt, flags=flags)

    def _afsirs_irrigation_props(self, gp, par):
        # irrigation parameters
        par(gp.add_text(name='afirr_label', label='Irrigation label', default=''))
        par(gp.add_float(name='wtd', label='Managed water table depth (<wtd>)', default=0.0, low=0.0, high=20.0))
        par(gp.add_boolean(name='netgross', label='Excess irrigation accounting (<netgross>)', default=False))
        par(gp.add_integer(name='irrmeth_id', label='Irrigation method id', default=-1))
        par(gp.add_float(name='eff', label='Irrigation application efficiency (<eff>)', default=0.0, low=0.0, high=1.0))
        lab = 'Fraction of area of parcel irrigated (<arzi>)'
        par(gp.add_float(name='arzi', label=lab, default=0.0, low=0.0, high=1.0))
        lab = 'Fraction of crop water use extracted from irrigated soil (<exir>)'
        par(gp.add_float(name='exir', label=lab, default=0.0, low=0.0, high=1.0))
        lab = 'Flood storage depth for rice, in (<drinc>)'
        par(gp.add_float(name='drinc', label=lab, default=0.0, low=0.0, high=12.0))
        par(gp.add_float(name='crown', label='Citrus bed height, in (<crown>)', default=0.0, low=0.0, high=36.0))
        par(gp.add_text(name='irrmgmt_label', label='Irrigation management label', default=''))

        lab = 'Irrigation trigger (trigcode)'
        opt_par = gp.add_option(name='trigcode', label=lab, default=IRR_TYPES[0], options=IRR_TYPES)
        par(opt_par)
        lab = 'Fixed irrigation rate, in/day (value)'
        p = gp.add_float(name='value_fixed', label=lab, default=0.0, low=0.0, high=2.0)
        flags = {IRR_OPT: False, IRR_FIX_RATE: True, IRR_FIX_DEF: False}
        p.add_dependency(parent=opt_par, flags=flags)
        p = gp.add_float(name='value_asw', label='Fraction of ASW (value)', default=0.0, low=0.0, high=100.0)
        flags = {IRR_OPT: False, IRR_FIX_RATE: False, IRR_FIX_DEF: True}
        p.add_dependency(parent=opt_par, flags=flags)

    def _afsirs_soil_props(self, gp, par, ag_opt_par, flags):
        # soil parameters
        par(gp.add_text(name='afsoil_label', label='Soil label', default=''))
        names = ['depth', 'minwc', 'maxwc']
        labels = [
            'Thickness of soil, in (depth)', 'Minimum soil water holding capacity (minwc)',
            'Maximum soil water holding capacity (maxwc)'
        ]
        highs = [96.0, 0.5, 1.0]
        _add_props(names, labels, highs, gp, ag_opt_par, flags)
        lab = 'Condition code for selecting available soil water capacity (cond)'
        par(gp.add_option(name='cond', label=lab, default=SOIL_COND_OPTS[0], options=SOIL_COND_OPTS))

    def _pumped_ditch_props(self, gp, ag_opt_par):
        flags = {t: False for t in AG_TYPES}
        flags[AG_TYPE_PD] = True
        names = [
            'percent_area', 'rks', 'fc_pump', 'ws_pump', 'bottom', 'max_level', 'min_level', 'fc_pump_on',
            'fc_pump_off', 'ws_pump_on', 'ws_pump_off'
        ]
        labels = [
            'Percent area of farm occupied by collector ditches (<perentarea>)',
            'Seepage coefficient (1/day) (rks)',
            'Pump rate for flood control pump (<fcPump>)',
            'Pump rate for water supply pump (wsPump)',
            'Bottom elevation of ditch (m-NAGD) (<bottom>)',
            'Trigger elevation for water supply pump turn on (m-NAGD) (<maxLevel>)',
            'Trigger elevation for water supply pump turn off (m-NAGD) (<minLevel>)',
            'Trigger elevation for flood control pump turn on (m-NAGD) (<fcPumpOn>)',
            'Trigger elevation for flood control pump turn off (m-NAGD) (<fcPumpOff>)',
            'Trigger elevation for water supply pump turn on (m-NAGD) (<wsPumpOn>)',
            'Trigger elevation for water supply pump turn off (m-NAGD) (<wsPumpOff>)',
        ]
        highs = [MAX_FLOAT] * len(names)
        highs[0] = 100.0  # percent area
        p = gp.add_text(name='ditch_runoff', label='Destination of runoff (<runoff>)', default='')
        p.add_dependency(parent=ag_opt_par, flags=flags)
        p = gp.add_boolean(name='bi_dir_seep', label='Bidirectional seepage (<biDirSeep>)', default=False)
        p.add_dependency(parent=ag_opt_par, flags=flags)
        _add_props(names, labels, highs, gp, ag_opt_par, flags)

    def _agimp_props(self, gp, ag_opt_par):
        flags = {t: False for t in AG_TYPES}
        flags[AG_TYPE_IMP] = True
        names = ['field_area', 'agimp_rks', 'height', 'r25y3d', 'allow', 's']
        labels = [
            'Area of the impoundment (<fieldArea>)', 'Seepage coefficient (1/day) (<rks>)',
            'Wet season water table elevation (<height>)',
            'Rainfall depth for 25-year, 3d return period storm (m) (<r25y3d>)',
            'Allowable basin discharge (m^3/s) (<allow>)', 'Abstraction in the NRCS runoff method (m) (<s>)'
        ]
        highs = [MAX_FLOAT] * len(names)
        p = gp.add_text(name='agimp_runoff', label='Destination of runoff (<runoff>)', default='')
        p.add_dependency(parent=ag_opt_par, flags=flags)
        _add_props(names, labels, highs, gp, ag_opt_par, flags)

    def _eaa_runoff_demand_props(self, gp, ag_opt_par):
        flags = {t: False for t in AG_TYPES}
        flags[AG_TYPE_EAA] = True
        names = [
            'kmax',
            'owpond',
            'drz',
            'srz',
        ]
        labels = [
            'Open water crop coefficient (<kmax>)',
            'Open water depth (ft) (<owpond>)',
            'Deep root zone depth (ft) (<drz>)',
            'Shallow root zone depth (ft) (<srz>)',
        ]
        highs = [MAX_FLOAT] * len(names)
        _add_props(names, labels, highs, gp, ag_opt_par, flags)
        p = gp.add_text(name='eaa_label', label='eaaRunoffDemand label (<label>)', default='')
        p.add_dependency(parent=ag_opt_par, flags=flags)
        p = gp.add_text(name='kcoef_id', label='Crop coefficient rulecurve ID (<kcoef_id>)', default='')
        p.add_dependency(parent=ag_opt_par, flags=flags)
        p = gp.add_text(name='kcalib_id', label='Calibration coefficient rulecurve ID (<kcalib_id>)', default='')
        p.add_dependency(parent=ag_opt_par, flags=flags)

    def load_ag_data(self, group: Group, hpm_data: HpmData):
        ag = hpm_data.ag
        ag.ag_type = group.parameter('ag_type').value
        self._load_prop[ag.ag_type](group, hpm_data)

    def _load_afsirs_data(self, group: Group, hpm_data: HpmData):
        af = hpm_data.ag.afsirs
        af.subday = group.parameter('subday').value
        af.coupled = group.parameter('coupled').value
        self._load_afsirs_crop_data(group, hpm_data)
        self._load_afsirs_irrigation_data(group, hpm_data)
        self._load_afsirs_soil_data(group, hpm_data)

    def _load_afsirs_crop_data(self, group: Group, hpm_data: HpmData):
        af = hpm_data.ag.afsirs
        af.crop_label = group.parameter('crop_label').value
        af.crop_id = group.parameter('crop_id').value
        af.j1 = group.parameter('j1').value
        af.jn = group.parameter('jn').value
        af.crop_type = group.parameter('crop_type').value
        if af.crop_type == CROP_PER:
            af.depth1 = group.parameter('depth1_per').value
            af.depth2 = group.parameter('depth2_per').value
            af.kctbl = group.parameter('kctbl_per').value
            af.awtbl = group.parameter('awtbl_per').value
        else:
            af.depth1 = group.parameter('depth1_ann').value
            af.depth2 = group.parameter('depth2_ann').value
            af.kctbl = group.parameter('kctbl_ann').value
            af.awtbl = group.parameter('awtbl_ann').value

    def _load_afsirs_irrigation_data(self, group: Group, hpm_data: HpmData):
        af = hpm_data.ag.afsirs
        af.afirr_label = group.parameter('afirr_label').value
        af.wtd = group.parameter('wtd').value
        af.netgross = group.parameter('netgross').value
        af.irrmeth_id = group.parameter('irrmeth_id').value
        af.eff = group.parameter('eff').value
        af.arzi = group.parameter('arzi').value
        af.exir = group.parameter('exir').value
        af.drinc = group.parameter('drinc').value
        af.crown = group.parameter('crown').value
        af.irrmgmt_label = group.parameter('irrmgmt_label').value
        af.trig_code = IRR_TYPES.index(group.parameter('trigcode').value)
        af.value_fixed = group.parameter('value_fixed').value
        af.value_asw = group.parameter('value_asw').value

    def _load_afsirs_soil_data(self, group: Group, hpm_data: HpmData):
        af = hpm_data.ag.afsirs
        af.soil_label = group.parameter('afsoil_label').value
        af.depth = group.parameter('depth').value
        af.minwc = group.parameter('minwc').value
        af.maxwc = group.parameter('maxwc').value
        af.cond = SOIL_COND_OPTS.index(group.parameter('cond').value) + 1

    def _load_pumped_ditch_data(self, group: Group, hpm_data: HpmData):
        pd = hpm_data.ag.pumped_ditch
        pd.percent_area = group.parameter('percent_area').value
        pd.rks = group.parameter('rks').value
        pd.fc_pump = group.parameter('fc_pump').value
        pd.ws_pump = group.parameter('ws_pump').value
        pd.bottom = group.parameter('bottom').value
        pd.max_level = group.parameter('max_level').value
        pd.min_level = group.parameter('min_level').value
        pd.fc_pump_on = group.parameter('fc_pump_on').value
        pd.fc_pump_off = group.parameter('fc_pump_off').value
        pd.ws_pump_on = group.parameter('ws_pump_on').value
        pd.ws_pump_off = group.parameter('ws_pump_off').value
        pd.runoff = group.parameter('ditch_runoff').value
        pd.bi_dir_seep = group.parameter('bi_dir_seep').value

    def _load_agimp_data(self, group: Group, hpm_data: HpmData):
        ai = hpm_data.ag.agimp
        ai.field_area = group.parameter('field_area').value
        ai.rks = group.parameter('agimp_rks').value
        ai.height = group.parameter('height').value
        ai.r25y3d = group.parameter('r25y3d').value
        ai.allow = group.parameter('allow').value
        ai.s = group.parameter('s').value
        ai.runoff = group.parameter('agimp_runoff').value

    def _load_eaa_runoff_demand_data(self, group: Group, hpm_data: HpmData):
        erd = hpm_data.ag.eaa_runoff_demand
        erd.kmax = group.parameter('kmax').value
        erd.owpond = group.parameter('owpond').value
        erd.drz = group.parameter('drz').value
        erd.srz = group.parameter('srz').value
        erd.kcoef_id = group.parameter('kcoef_id').value
        erd.kcalib_id = group.parameter('kcalib_id').value
        erd.label = group.parameter('eaa_label').value


class _HpmUrbanDataDef:
    """A class to define the urban data for the generic model."""
    def __init__(self):
        self._load_prop = {
            URBAN_TYPE_IMPERV: self._load_imperv,
            URBAN_TYPE_MBRCELL: self._load_mbrcell,
            URBAN_TYPE_CU: self._load_cu,
            URBAN_TYPE_URBANDET: self._load_urbandet
        }

    def _make_flags(self, the_type):
        flags = {t: False for t in URBAN_TYPES}
        flags[the_type] = True
        return flags

    def _imperv(self, gp, opt_par):
        flags = self._make_flags(URBAN_TYPE_IMPERV)
        lab = 'Directly or indirectly connected (<dirconn>)'
        d_opt = gp.add_option(name='imperv_dirconn', label=lab, default=DIRCON_TYPES[0], options=DIRCON_TYPES)
        d_opt.add_dependency(parent=opt_par, flags=flags)
        flags = self._make_flags(URBAN_TYPE_IMPERV)
        names = ['isto', 'isto1', 'sdet', 'sdet1']
        names = [f'imperv_{n}' for n in names]
        labels = [
            'Maximum interception storage depth (m) (<isto>)', 'Initial interception storage depth (m) (<isto1>)',
            'Maximum depressional surface storage depth (m) (<sdet>)',
            'Initial depressional surface storage depth (m) (<sdet1>)'
        ]
        highs = [0.1, 0.1, 0.2, 0.2]
        _add_props(names, labels, highs, gp, opt_par, flags)

    def _mbrcell(self, gp, opt_par):
        flags = self._make_flags(URBAN_TYPE_MBRCELL)
        txt_par = gp.add_text(name='mbrcell_route', label='Label of waterbody for discharge (<route>)', default='')
        txt_par.add_dependency(parent=opt_par, flags=flags)
        lab = 'Time of concentration (s) (<tc>)'
        ipar = gp.add_integer(name='mbrcell_tc', label=lab, default=900, low=900, high=500000)
        ipar.add_dependency(parent=opt_par, flags=flags)
        names = ['kveg', 'd_shal', 'd_deep', 'fld_cap']
        names = [f'mbrcell_{n}' for n in names]
        labels = [
            'Vegetation crop coefficient (<kveg>)', 'Shallow root zone depth (<d_shal>)',
            'Deep root zone depth (<d_deep>)', 'Water content at field capacity (<fld_cap>)'
        ]
        highs = [1.0, 3.0, 5.0, 2.0]
        _add_props(names, labels, highs, gp, opt_par, flags)

    def _cu(self, gp, opt_par):
        flags = self._make_flags(URBAN_TYPE_CU)
        lab = 'High density (HI) or low density (LI) landuse (<label>)'
        cu_opt = gp.add_option(name='cu_label', label=lab, default=CU_TYPES[0], options=CU_TYPES)
        cu_opt.add_dependency(parent=opt_par, flags=flags)
        lab = 'Percent area of the Hub with the specified consumptive use (<percentarea>)'
        f_par = gp.add_float(name='cu_percentarea', label=lab, default='0.0', high=100.0)
        f_par.add_dependency(parent=opt_par, flags=flags)
        txt_par = gp.add_text(name='cu_wsupply', label='Water supply source waterbody label (<wsupply>)', default='')
        txt_par.add_dependency(parent=opt_par, flags=flags)
        txt_par = gp.add_text(name='cu_sewer', label='Sewer destination waterbody label (<sewer>)', default='')
        txt_par.add_dependency(parent=opt_par, flags=flags)
        lab = 'Trigger category (<trigcat>)'
        cu_opt = gp.add_option(name='cu_trigcat', label=lab, default=CU_TRIG_TYPES[0], options=CU_TRIG_TYPES)
        cu_opt.add_dependency(parent=opt_par, flags=flags)
        lab = 'Consumptive use value (<const value>)'
        f_par = gp.add_float(name='cu_const_value', label=lab, default=0.0, low=0.0)
        f_par.add_dependency(parent=opt_par, flags=flags)
        lab = 'Element under <sewer> of fraction loss to groundwater (<fracloss>)'
        f_par = gp.add_float(name='cu_fracloss', label=lab, default=0.0, low=0.0, high=1.0)
        f_par.add_dependency(parent=opt_par, flags=flags)
        lab = 'Discharge water from <cu> goes to home waterbody rather than directed waterbody (<septic>)'
        b_par = gp.add_boolean(name='cu_septic', label=lab, default=False)
        b_par.add_dependency(parent=opt_par, flags=flags)

    def _urbanet(self, gp, opt_par):
        flags = self._make_flags(URBAN_TYPE_URBANDET)
        lab = 'Runoff destination waterbody label (<runoff>)'
        txt_par = gp.add_text(name='urbandet_runoff', label=lab, default='')
        txt_par.add_dependency(parent=opt_par, flags=flags)
        names = ['percentarea', 'rks', 'wlen', 'angle', 'top', 'apx']
        names = [f'urbandet_{n}' for n in names]
        labels = [
            'Percent area of the parcel in the impoundment <percentarea>)', 'Seepage coefficient (<rks>)',
            'Length of rectangular weir (m) (<wlen>)', 'Angle of v-notch weir (degrees) (<angle>)',
            'Elevation of top of v-notch bleeder (m) (<top>)', 'Elevation of invert of v-notch bleeder (m) (<apx>)'
        ]
        highs = [25.0, 0.1, 10.0, 120.0, 20.0, 20.0]
        _add_props(names, labels, highs, gp, opt_par, flags)

    def urban_props(self, gp, hpm_type_par):
        hpm_type_flags = {t: False for t in HPM_TYPES}
        hpm_type_flags[HPM_TYPE_URBAN] = True
        opt_par = gp.add_option(name='urban_type', label='Urban type', default=URBAN_TYPES[0], options=URBAN_TYPES)
        opt_par.add_dependency(parent=hpm_type_par, flags=hpm_type_flags)
        self._imperv(gp, opt_par)
        self._mbrcell(gp, opt_par)
        self._cu(gp, opt_par)
        self._urbanet(gp, opt_par)

    def load_urban_data(self, group: Group, hpm_data: HpmData):
        urban = hpm_data.urban
        urban.urban_type = group.parameter('urban_type').value
        self._load_prop[urban.urban_type](group, hpm_data)

    def _load_imperv(self, group: Group, hpm_data: HpmData):
        imp = hpm_data.urban.imperv
        imp.dirconn = 1
        if group.parameter('imperv_dirconn').value == DIRCON_0:
            imp.dirconn = 0
        imp.isto = group.parameter('imperv_isto').value
        imp.isto1 = group.parameter('imperv_isto1').value
        imp.sdet = group.parameter('imperv_sdet').value
        imp.sdet1 = group.parameter('imperv_sdet1').value

    def _load_mbrcell(self, group: Group, hpm_data: HpmData):
        mbr = hpm_data.urban.mbrcell
        mbr.route = group.parameter('mbrcell_route').value
        mbr.tc = group.parameter('mbrcell_tc').value
        mbr.kveg = group.parameter('mbrcell_kveg').value
        mbr.d_shal = group.parameter('mbrcell_d_shal').value
        mbr.d_deep = group.parameter('mbrcell_d_deep').value
        mbr.fld_cap = group.parameter('mbrcell_fld_cap').value

    def _load_cu(self, group: Group, hpm_data: HpmData):
        cu = hpm_data.urban.cu
        cu.label = group.parameter('cu_label').value
        cu.percentarea = group.parameter('cu_percentarea').value
        cu.wsupply = group.parameter('cu_wsupply').value
        cu.sewer = group.parameter('cu_sewer').value
        cu.trigcat = group.parameter('cu_trigcat').value
        cu.const_value = group.parameter('cu_const_value').value
        cu.fracloss = group.parameter('cu_fracloss').value
        cu.septic = group.parameter('cu_septic').value

    def _load_urbandet(self, group: Group, hpm_data: HpmData):
        urbandet = hpm_data.urban.urbandet
        urbandet.runoff = group.parameter('urbandet_runoff').value
        urbandet.percentarea = group.parameter('urbandet_percentarea').value
        urbandet.rks = group.parameter('urbandet_rks').value
        urbandet.wlen = group.parameter('urbandet_wlen').value
        urbandet.angle = group.parameter('urbandet_angle').value
        urbandet.top = group.parameter('urbandet_top').value
        urbandet.apx = group.parameter('urbandet_apx').value


class _HpmDataDef:
    """A class to define the HPM data for the generic model."""
    def natural_system_props(self, gp, hpm_type_par=None):
        ns_props = _HpmNaturalSystemDataDef()
        ns_props.natural_system_props(gp, hpm_type_par)

    def agricultural_props(self, gp, hpm_type_par=None):
        ag_props = _HpmAgriculturalDataDef()
        ag_props.agricultural_props(gp, hpm_type_par)

    def urban_props(self, gp, hpm_type_par=None):
        urban = _HpmUrbanDataDef()
        urban.urban_props(gp, hpm_type_par)

    def load_data_from_group(self, group: Group, hpm_data: HpmData):
        """Loads the data from the group into the HpmData object.

        Args:
            group (Group): The group to load the data from.
            hpm_data (HpmData): The HpmData object to load the data into.
        """
        hpm_data.hpm_type = group.parameter('hpm_type').value
        if hpm_data.hpm_type == HPM_TYPE_NONE:
            return
        elif hpm_data.hpm_type == HPM_TYPE_NS:
            ns_prop = _HpmNaturalSystemDataDef()
            ns_prop.load_ns_data(group, hpm_data)
        elif hpm_data.hpm_type == HPM_TYPE_AG:
            ag_prop = _HpmAgriculturalDataDef()
            ag_prop.load_ag_data(group, hpm_data)
        else:  # hpm_data.hpm_type == HPM_TYPE_URBAN:
            urban_prop = _HpmUrbanDataDef()
            urban_prop.load_urban_data(group, hpm_data)


def _add_props(names, labels, highs, gp, opt_par, flags):
    for name, label, high in zip(names, labels, highs):
        p = gp.add_float(name=name, label=label, default=0.0, low=0.0, high=high)
        p.add_dependency(parent=opt_par, flags=flags)


def load_data_from_group(group: Group, hpm_data: HpmData):
    """Loads the HPM data from the group into the HpmData object.

    Args:
        group (Group): The group to load the data from.
        hpm_data (HpmData): The HpmData object to load the data into.
    """
    dd = _HpmDataDef()
    dd.load_data_from_group(group, hpm_data)


def add_hpm_to_section(section: Section, group_name='', group_label='', include_index=True):
    """Adds the HPM polygon parameters to the section.

    Args:
        section (Section): The section to add the HPM polygon parameters to.
        group_name (str, optional): The name of the new group. Defaults to 'hpm'.
        group_label (str, optional): The label of the new group. Defaults to 'Hydraulic Process Modules'.
        include_index (bool, optional): Whether to include the index parameter in the new group.
    """
    g_name = group_name if group_name else 'hpm'
    g_label = group_label if group_label else 'Hydraulic Process Modules'
    gp = section.add_group(group_name=g_name, label=g_label)
    gp.add_text(name='label', label='Label')
    if include_index:
        spec_par = gp.add_boolean(name='specify_index', label='Specify index', default=False)
        p = gp.add_integer(name='index_value', label='Index (values <= 0 ignored)', default=0)
        flags = {True: True, False: False}
        p.add_dependency(parent=spec_par, flags=flags)
    hpm_type_par = gp.add_option(name='hpm_type', label='HPM type', default=HPM_TYPES[0], options=HPM_TYPES)
    dd = _HpmDataDef()
    dd.natural_system_props(gp, hpm_type_par=hpm_type_par)
    dd.agricultural_props(gp, hpm_type_par=hpm_type_par)
    dd.urban_props(gp, hpm_type_par=hpm_type_par)
