"""Mesh data definition."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel, Group, Section
from xms.guipy.widgets.table_with_tool_bar import FloatColumnType, TableDefinition

# 4. Local modules
import xms.rsm.data.hpm_data_def as hpm_dd

OPT_NOT_SPECIFIED = 'Not specified'
OPT_CONSTANT = 'Constant (<const>)'
OPT_CURVE = 'Curve (<csv>)'
OPT_DATASET = 'Dataset (<gms>)'
OPT_DSS = 'DSS file (<dss>)'
OPT_GRID_FILE = 'Grid file (<gridio>)'
OPT_NETCDF_FILE = 'NetCDF file (<netcdf>)'
OPT_IDX_DATASET = 'Dataset (<index>)'
OPT_TR_LOOKUP = 'Lookup table (<lookuptr>)'
OPT_CONVEY_MANNING = 'Mannings (<mannings>)'
OPT_CONVEY_KADLEC = 'Kadlec (<kadlec>)'
OPT_CONVEY_LOOKUP = 'Lookup table (<lookup>)'
OPT_SV_CONSTANT = 'Constant (<constsv>)'
OPT_SV_DATASET = 'Dataset (<layersv>)'
OPT_SV_LOOKUP = 'Lookup table (<lookupsv>)'

PROP_SURFACE = 'surface'
PROP_BOTTOM = 'bottom'
PROP_SHEAD = 'shead'
PROP_TRANSMISSIVITY = 'transmissivity'
PROP_VERT_CONDUCTIVITY = 'vert_conductivity'
PROP_RAIN = 'rain'
PROP_REFET = 'refet'
PROP_CONVEYANCE = 'conveyance'
PROP_SVCONVERTER = 'svconverter'
PROP_HPM = 'hpm'
PROP_LIST = [
    PROP_SURFACE, PROP_BOTTOM, PROP_SHEAD, PROP_TRANSMISSIVITY, PROP_VERT_CONDUCTIVITY, PROP_RAIN, PROP_REFET,
    PROP_CONVEYANCE, PROP_SVCONVERTER, PROP_HPM
]


class _MeshGroupDataAdder:
    """Helper class to hold group information."""
    def __init__(self, group, label, add_layer):
        """Constructor.

        Args:
            group (xms.gmi.data.generic_model.Group): the generic model group
            label (str): label of the group
            add_layer (bool): True if the group should have a layer
        """
        self.group = group
        self.label = label
        self.add_layer = add_layer
        self.add_specify_index = True
        self.add_spatial_inputs = True
        self.opts = [OPT_NOT_SPECIFIED, OPT_CONSTANT, OPT_DATASET]
        self._opt_const = OPT_CONSTANT
        self._opt_ds = OPT_DATASET
        self._opt_par = None
        self._const_value_xml_tag = '<value>'
        self._ds_xml_tags = '<gms>, <gmslayer>'

    def _add_const_value_to_group(self):
        """Add constant value to the group."""
        gm_gp, parent, opts = self.group, self._opt_par, self.opts
        const_flags = {opt: False for opt in opts}
        const_flags[self._opt_const] = True
        tag = self._const_value_xml_tag
        cv = gm_gp.add_float(name='constant_value', label=f'Constant value ({tag})', default=0.0)
        cv.add_dependency(parent, const_flags)
        cv = gm_gp.add_float(name='constant_value_mult', label='Multipler (<mult>)', default=1.0)
        cv.add_dependency(parent, const_flags)

    def _add_specify_index_to_group(self):
        """Add fields to specify index for this property."""
        if not self.add_specify_index:
            return
        gm_gp = self.group
        tog = gm_gp.add_boolean(name='specify_index', label='Specify index', default=False)
        index = gm_gp.add_integer(name='index_value', label='Index value', default=1, low=1)
        index.add_dependency(parent=tog, flags={True: True, False: False})

    def add_const_dataset_fields(self):
        """Add fields to the group."""
        gm_gp, opts = self.group, self.opts
        gm_gp.add_text(name='label', label='Label', default='')
        self._add_specify_index_to_group()
        self._opt_par = gm_gp.add_option(name='input_type', label=self.label, options=opts, default=opts[0])
        self._add_const_value_to_group()
        if not self.add_spatial_inputs:
            return
        # dataset
        ds_flags = {opt: False for opt in opts}
        ds_flags[self._opt_ds] = True
        tags = self._ds_xml_tags
        ds = gm_gp.add_dataset(name='dataset', label=f'Dataset ({tags})')
        ds.add_dependency(parent=self._opt_par, flags=ds_flags)
        ds_mult = gm_gp.add_float(name='dataset_mult', label='Dataset multiplier (<mult>)', default=1.0)
        ds_mult.add_dependency(parent=self._opt_par, flags=ds_flags)
        if self.add_layer:
            lay = gm_gp.add_boolean(name='specify_layer', label='Specify layer', default=False)
            lay.add_dependency(parent=self._opt_par, flags=ds_flags)
            lay_val = gm_gp.add_integer(name='layer', label='Layer (<layer>)', default=1, low=1)
            lay_val.add_dependency(parent=lay, flags={True: True, False: False})

    def _add_transmissivity_lookup(self):
        """Add lookup table to the transmissivity group."""
        gm_gp, parent, opts = self.group, self._opt_par, self.opts
        tab_flags = {opt: False for opt in opts}
        tab_flags[OPT_TR_LOOKUP] = True
        items = ['below', 'above']
        for item in items:
            lp = gm_gp.add_float(name=f'lookup_{item}', label=f'Lookup table "{item}" (<{item}>)', default=0.0)
            lp.add_dependency(parent=parent, flags=tab_flags)
        table_def = TableDefinition(
            [FloatColumnType(header='Head', default=0.0),
             FloatColumnType(header='Transmissivity', default=0.0)],
        )
        default_vals = []
        label = 'Lookup table (<lookuptr>)'
        lp = gm_gp.add_table(name='lookup_table', label=label, table_definition=table_def, default=default_vals)
        lp.add_dependency(parent=parent, flags=tab_flags)

    def add_transmissivity_fields(self):
        """Add the transmissivity group to the GenericModel class."""
        gm_gp = self.group
        self.opts = [OPT_NOT_SPECIFIED, OPT_CONSTANT, OPT_DATASET, OPT_TR_LOOKUP]
        gm_gp.add_boolean(name='unconfined', label='Unconfined (<unconfined>)', default=False)
        self._ds_xml_tags = '<confined_gms>, <confined_gms_layer>, <unconfined_gms>, <unconfined_gms_layer>'
        self.add_const_dataset_fields()
        self._add_transmissivity_lookup()

    def _add_csv_to_rain_et(self):
        """Add csv to the rain/evapotranspiration group."""
        gm_gp, parent, opts = self.group, self._opt_par, self.opts
        # csv - curve
        csv_flags = {opt: False for opt in opts}
        csv_flags[OPT_CURVE] = True
        label = 'Time series curve (<csv>)'
        csv = gm_gp.add_curve(name='csv', label=label, use_dates=True, axis_titles=['Time', 'Value'])
        csv.add_dependency(parent=parent, flags=csv_flags)

    def _add_dss_to_rain_et(self):
        """Add dss to the rain/evapotranspiration group."""
        gm_gp, parent, opts = self.group, self._opt_par, self.opts
        # dss file, pn(str), mult(float)
        file_filter = 'DSS files (*.dss);;All files (*)'
        dssp = gm_gp.add_input_file(name='dss_file', label='DSS file (<file>)', default='', file_filter=file_filter)
        dss_flags = {opt: False for opt in opts}
        dss_flags[OPT_DSS] = True
        dssp.add_dependency(parent=parent, flags=dss_flags)
        dssp = gm_gp.add_text(name='dss_path', label='DSS path (<pn>)', default='')
        dssp.add_dependency(parent=parent, flags=dss_flags)
        dssp = gm_gp.add_float(name='dss_mult', label='DSS multiplier (<mult>)', default=1.0)
        dssp.add_dependency(parent=parent, flags=dss_flags)
        dssp = gm_gp.add_text(name='dss_units', label='DSS units (<units>)', default='')
        dssp.add_dependency(parent=parent, flags=dss_flags)
        opts = [OPT_NOT_SPECIFIED, 'INST-VAL', 'INST-CUM', 'PER-CUM', 'PER-AVE']
        plabel = 'DSS type of data (<type>)'
        dssp = gm_gp.add_option(name='dss_type', label=plabel, default=opts[0], options=opts)
        dssp.add_dependency(parent=parent, flags=dss_flags)

    def _add_gridio_to_rain_et(self):
        """Add gridio to the rain/evapotranspiration group."""
        gm_gp, parent, opts = self.group, self._opt_par, self.opts
        # grid - file, dbintl(int), mult(float), x origin(float), y origin(float)
        file_filter = 'GridIO files (*.bin);;All files (*)'
        gfp = gm_gp.add_input_file(name='grid_file', label='Grid file (<file>)', default='', file_filter=file_filter)
        grid_flags = {opt: False for opt in opts}
        grid_flags[OPT_GRID_FILE] = True
        gfp.add_dependency(parent=parent, flags=grid_flags)
        gfp = gm_gp.add_float(name='grid_file_x', label='X origin (<xorig>)', default=0.0)
        gfp.add_dependency(parent=parent, flags=grid_flags)
        gfp = gm_gp.add_float(name='grid_file_y', label='Y origin (<yorig>)', default=0.0)
        gfp.add_dependency(parent=parent, flags=grid_flags)
        gfp = gm_gp.add_float(name='grid_file_mult', label='Grid file multiplier (<mult>)', default=1.0)
        gfp.add_dependency(parent=parent, flags=grid_flags)
        gfp = gm_gp.add_integer(name='grid_file_dbintl', label='Grid file database interval (<dbintl>)', default=0)
        gfp.add_dependency(parent=parent, flags=grid_flags)

    def _add_netcdf_to_rain_et(self):
        """Add netcdf to the rain/evapotranspiration group."""
        gm_gp, parent, opts = self.group, self._opt_par, self.opts
        # netcdf - file, variable(str), dbintl(int), mult(float), units(str)
        file_filter = 'NetCDF files (*.nc);;All files (*)'
        label = 'NetCDF file (<file>)'
        ncp = gm_gp.add_input_file(name='netcdf_file', label=label, default='', file_filter=file_filter)
        ncd_flags = {opt: False for opt in opts}
        ncd_flags[OPT_NETCDF_FILE] = True
        ncp.add_dependency(parent=parent, flags=ncd_flags)
        ncp = gm_gp.add_text(name='netcdf_variable', label='NetCDF variable (<variable>)', default='')
        ncp.add_dependency(parent=parent, flags=ncd_flags)
        ncp = gm_gp.add_float(name='netcdf_mult', label='NetCDF multiplier (<mult>)', default=1.0)
        ncp.add_dependency(parent=parent, flags=ncd_flags)
        ncp = gm_gp.add_integer(name='netcdf_dbintl', label='NetCDF database interval (<dbintl>)', default=0)
        ncp.add_dependency(parent=parent, flags=ncd_flags)
        ncp = gm_gp.add_text(name='netcdf_units', label='NetCDF units (<units>)', default='')
        ncp.add_dependency(parent=parent, flags=ncd_flags)

    def add_rain_et_fields(self):
        """Add the rain/evapotranspiration group to the GenericModel class."""
        self.opts = [OPT_NOT_SPECIFIED, OPT_CONSTANT, OPT_CURVE, OPT_DATASET, OPT_DSS, OPT_GRID_FILE, OPT_NETCDF_FILE]
        if not self.add_spatial_inputs:
            self.opts = [OPT_NOT_SPECIFIED, OPT_CONSTANT, OPT_CURVE, OPT_DSS, OPT_GRID_FILE]
        self.add_const_dataset_fields()
        self._add_csv_to_rain_et()  # csv
        self._add_dss_to_rain_et()  # DSS
        self._add_gridio_to_rain_et()  # gridio
        if self.add_spatial_inputs:
            self._add_netcdf_to_rain_et()  # netcdf

    def _add_conveyance_mannings(self):
        """Add the mannings coefficients to the conveyance group."""
        gm_gp, parent, opts = self.group, self._opt_par, self.opts
        # manning coefficients
        coefs = ['a', 'b', 'detent']
        man_flags = {opt: False for opt in opts}
        man_flags[OPT_CONVEY_MANNING] = True
        for coef in coefs:
            label = f'Specify mannings "{coef}" coefficient'
            tog = gm_gp.add_boolean(name=f'manning_{coef}', label=label, default=False)
            tog.add_dependency(parent=parent, flags=man_flags)
            label = f'Mannings "{coef}" coefficient (<mannings {coef}>)'
            flt = gm_gp.add_float(name=f'manning_{coef}_value', label=label, default=0.0)
            flt.add_dependency(parent=tog, flags={True: True, False: False})

    def _add_conveyance_kadlec(self):
        """Add kadlec coefficients to the conveyance group."""
        gm_gp, parent, opts = self.group, self._opt_par, self.opts
        # kadlec coefficients
        coefs = ['K', 'alpha', 'beta', 'detent']
        k_flags = {opt: False for opt in opts}
        k_flags[OPT_CONVEY_KADLEC] = True
        for coef in coefs:
            label = f'Specify kadlec "{coef}" coefficient'
            tog = gm_gp.add_boolean(name=f'kadlec_{coef}', label=label, default=False)
            tog.add_dependency(parent=parent, flags=k_flags)
            label = f'Kadlec "{coef}" coefficient (<kadlec {coef}>)'
            flt = gm_gp.add_float(name=f'kadlec_{coef}_value', label=label, default=0.0)
            flt.add_dependency(parent=tog, flags={True: True, False: False})

    def _add_conveyance_lookup(self):
        """Add lookup table to the conveyance group."""
        gm_gp, parent, opts = self.group, self._opt_par, self.opts
        tab_flags = {opt: False for opt in opts}
        tab_flags[OPT_CONVEY_LOOKUP] = True
        items = ['below', 'above', 'base', 'exponent']
        for item in items:
            lp = gm_gp.add_float(name=f'lookup_{item}', label=f'Lookup table "{item}" (<{item}>)', default=0.0)
            lp.add_dependency(parent=parent, flags=tab_flags)
        table_def = TableDefinition(
            [FloatColumnType(header='Depth', default=0.0),
             FloatColumnType(header='Conveyance', default=0.0)],
        )
        default_vals = []
        label = 'Lookup table (<lookup>)'
        lp = gm_gp.add_table(name='lookup_table', label=label, table_definition=table_def, default=default_vals)
        lp.add_dependency(parent=parent, flags=tab_flags)

    def add_conveyance_fields(self):
        """Add conveyance fields to the GenericModel class."""
        gm_gp = self.group
        gm_gp.add_text(name='label', label='Label', default='')
        self._add_specify_index_to_group()
        self.opts = opts = [OPT_NOT_SPECIFIED, OPT_CONVEY_MANNING, OPT_CONVEY_KADLEC, OPT_CONVEY_LOOKUP]
        self._opt_par = gm_gp.add_option(name='input_type', label=self.label, options=opts, default=opts[0])
        self._add_conveyance_mannings()  # manning coefficients
        self._add_conveyance_kadlec()  # kadlec coefficients
        self._add_conveyance_lookup()  # lookup table

    def _add_svconverter_lookup(self):
        """Add lookup table to the stage/volume converter group."""
        gm_gp, parent, opts = self.group, self._opt_par, self.opts
        tab_flags = {opt: False for opt in opts}
        tab_flags[OPT_SV_LOOKUP] = True
        items = ['sc', 'below', 'above']
        for item in items:
            lp = gm_gp.add_float(name=f'lookup_{item}', label=f'Lookup table "{item}" (<{item}>)', default=0.0)
            lp.add_dependency(parent=parent, flags=tab_flags)
        table_def = TableDefinition(
            [FloatColumnType(header='Stage', default=0.0),
             FloatColumnType(header='Volume', default=0.0)],
        )
        default_vals = []
        label = 'Lookup table (<lookupsv>)'
        lp = gm_gp.add_table(name='lookup_table', label=label, table_definition=table_def, default=default_vals)
        lp.add_dependency(parent=parent, flags=tab_flags)

    def add_svconverter_fields(self):
        """Add stage/volume converter fields to the GenericModel class."""
        self.opts = [OPT_NOT_SPECIFIED, OPT_SV_CONSTANT, OPT_SV_DATASET, OPT_SV_LOOKUP]
        self._opt_const = OPT_SV_CONSTANT
        self._opt_ds = OPT_SV_DATASET
        # self._opt_par = gm_gp.add_option(name='input_type', label=self.label, options=opts, default=opts[0])
        self._const_value_xml_tag = '<sc>'
        self._ds_xml_tags = '<layersv>'
        self.add_const_dataset_fields()
        self._add_svconverter_lookup()


class _MeshGroupDataDef:
    """Helper class to hold group information."""
    def __init__(self, section):
        """Initializes the class.

        Args:
            section (xms.gmi.data.generic_model.Section): the section to add the mesh data parameters to
        """
        self.section = section

    def add_groups(self):
        """Add the groups to the GenericModel class."""
        self._add_surface()
        self._add_bottom()
        self._add_shead()
        self._add_transmissivity()
        self._add_vert_conductivity()
        self._add_rain()
        self._add_refet()
        self._add_conveyance()
        self._add_svconverter()
        self._add_hpm()

    def _add_group_with_const_and_dataset(self, group_name, group_label, data_label):
        """Add fields to a group in the GenericModel class.

        Args:
            group_name (str): name of the group
            group_label (str): label for the group
            data_label (str): label for the data
        """
        grp = self.section.add_group(group_name=group_name, label=group_label)
        da = _MeshGroupDataAdder(group=grp, label=data_label, add_layer=True)
        da.add_const_dataset_fields()

    def _add_surface(self):
        """Add <surface>, Ground surface elevation to the GenericModel class."""
        # <surface>: <indexed>, <const>, <gmslayer>, <gms>
        group_name, group_label = 'surface', 'Ground surface elevation'
        data_label = 'Ground surface elevation (<surface>)'
        self._add_group_with_const_and_dataset(group_name, group_label, data_label)

    def _add_bottom(self):
        """Add <bottom>, Bottom elevation to the GenericModel class."""
        # <bottom>: <indexed>, <const>, <gmslayer>, <gms>
        group_name, group_label = 'bottom', 'Bottom elevation'
        data_label = 'Bottom elevation (<bottom>)'
        self._add_group_with_const_and_dataset(group_name, group_label, data_label)

    def _add_shead(self):
        """Add <shead>, Starting heads to the GenericModel class."""
        # <shead>: <indexed>, <const>, <gmslayer>, <gms>
        group_name, group_label = 'shead', 'Starting heads'
        data_label = 'Starting heads (<shead>)'
        self._add_group_with_const_and_dataset(group_name, group_label, data_label)

    def _add_transmissivity(self):
        """Add <transmissivity>, Transmissivity to the GenericModel class."""
        # <transmissivity>:
        # <indexed>, <confined>, <confined_gms>, <confined_gms_layer>, <layered>, <layered_gms_layer>,
        # <lookuptr>, <unconfined>, <unconfined_gms>, <unconfined_gms_layer>
        group_name, group_label = 'transmissivity', 'Transmissivity'
        data_label = 'Transmissivity (<transmissivity>)'
        grp = self.section.add_group(group_name=group_name, label=group_label)
        da = _MeshGroupDataAdder(group=grp, label=data_label, add_layer=True)
        da.add_transmissivity_fields()

    def _add_vert_conductivity(self):
        """Add <vert_conductivity>, Vertical conductivity to the GenericModel class."""
        # <vert_conductivity>: <indexed>, <const>, <gmslayer>, <gms>
        group_name, group_label = 'vert_conductivity', 'Vertical conductivity'
        data_label = 'Vertical conductivity (<vert_conductivity>)'
        self._add_group_with_const_and_dataset(group_name, group_label, data_label)

    def _add_rain_et_group(self, group_name, group_label, data_label):
        """Add fields to a group in the GenericModel class.

        Args:
            group_name (str): name of the group
            group_label (str): label for the group
            data_label (str): label for the data
        """
        grp = self.section.add_group(group_name=group_name, label=group_label)
        da = _MeshGroupDataAdder(group=grp, label=data_label, add_layer=False)
        da.add_rain_et_fields()

    def _add_rain(self):
        """Add <rain>, Rain to the GenericModel class."""
        # <rain>: <dss>, <indexed>, <const>, <note>, <gridio>, <tsin>, <netcdf>, <csv>
        group_name, group_label = 'rain', 'Rain'
        data_label = 'Rain (<rain>)'
        self._add_rain_et_group(group_name, group_label, data_label)

    def _add_refet(self):
        """Add <refet>, Evapotranspiration to the GenericModel class."""
        group_name, group_label = 'refet', 'Evapotranspiration'
        data_label = 'Evapotranspiration (<refet>)'
        self._add_rain_et_group(group_name, group_label, data_label)

    def _add_conveyance(self):
        """Add <conveyance>, Conveyance to the GenericModel class."""
        # <conveyance>: <indexed>, <mannings>, <kadlec>, <lookup>
        group_name, group_label = 'conveyance', 'Conveyance'
        data_label = 'Conveyance (<conveyance>)'
        grp = self.section.add_group(group_name=group_name, label=group_label)
        da = _MeshGroupDataAdder(group=grp, label=data_label, add_layer=False)
        da.add_conveyance_fields()

    def _add_svconverter(self):
        """Add <svconverter>, Stage/Volume converter to the GenericModel class."""
        # <svconverter>: <indexed>, <constsv>, <layersv>, <lookupsv>
        group_name, group_label = 'svconverter', 'Stage/Volume converter'
        data_label = 'Stage/Volume converter (<svconverter>)'
        grp = self.section.add_group(group_name=group_name, label=group_label)
        da = _MeshGroupDataAdder(group=grp, label=data_label, add_layer=True)
        da.add_svconverter_fields()

    def _add_hpm(self):
        """Add HPM to the GenericModel class."""
        hpm_dd.add_hpm_to_section(self.section)


def generic_model():
    """Gets a generic model for the boundary conditions coverage.

    Returns:
        (xms.gmi.data.generic_model.GenericModel): the generic model class
    """
    gm = GenericModel()
    md = _MeshGroupDataDef(section=gm.polygon_parameters)
    md.add_groups()
    return gm


def add_mesh_data_to_section(section: Section):
    """Add the mesh data parameters to the section.

    Args:
        section (xms.gmi.data.generic_model.Section): the section to add the mesh data parameters to
    """
    md = _MeshGroupDataDef(section=section)
    md.add_groups()


def add_rain_et_to_group(rain_grp: Group, et_grp: Group):
    """Add the rain/evapotranspiration parameters to the group.

    Args:
        rain_grp (xms.gmi.data.generic_model.Group): the group to add the rain parameters to
        et_grp (xms.gmi.data.generic_model.Group): the group to add the evapotranspiration parameters to
    """
    da = _MeshGroupDataAdder(group=rain_grp, label='Rain (<rain>)', add_layer=False)
    da.add_specify_index = False
    da.add_spatial_inputs = False
    da.add_rain_et_fields()
    da = _MeshGroupDataAdder(group=et_grp, label='Evapotranspiration (<refet>)', add_layer=False)
    da.add_specify_index = False
    da.add_spatial_inputs = False
    da.add_rain_et_fields()
