"""Module for exporting an RSM simulation with feedback."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Signal

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg, ProcessFeedbackThread
from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists

# 4. Local modules
from xms.rsm.dmi.xms_data import XmsData
from xms.rsm.file_io import util
from xms.rsm.file_io.control_writer import ControlWriter


class ExportSimulationRunner(ProcessFeedbackThread):
    """Class for exporting an RSM simulation in a worker thread."""
    processing_finished = Signal()

    def __init__(self, parent):
        """Constructor.

        Args:
            parent (QWidget): Parent of the QThread, probably the hidden parent dialog created by XMS.
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self._xms_data = XmsData()
        self._mesh_file_name = None
        self._logger = util.get_logger()
        self._exported_sim_uuids = set()

    def _export_simulation(self):
        """Export a simulation.

        Args:
            query (Query): XMS interprocess communicator
            sim_uuid (str): UUID of the simulation to export, pass empty string for the root simulation
            sim_name (str): Name of the simulation to export, pass empty string for the root simulation
        """
        self._logger.info('Establishing communication with SMS.')
        writer = ControlWriter(self._xms_data)
        writer.write()

    def _do_work(self):
        """Thread worker method."""
        try:
            self._export_simulation()
        except Exception:
            self._logger.error('Error exporting simulation.')


def export_simulation_with_feedback(runner=None):
    """Export a simulation with a feedback dialog.

    Args:
        runner (ExportSimulationRunner): exporter class
    """
    ensure_qapplication_exists()
    worker = ExportSimulationRunner(None) if runner is None else runner

    display_text = {
        'title': 'RSM Export Simulation',
        'working_prompt': 'Exporting RSM simulation files. Please wait...',
        'warning_prompt': 'Warning(s) encountered while exporting simulation. Review log output for more details.',
        'error_prompt': 'Error(s) encountered while exporting simulation. Review log output for more details.',
        'success_prompt': 'Successfully exported simulation',
        'note': '',
        'auto_load': 'Close this dialog automatically when exporting is finished.'
    }
    feedback_dlg = ProcessFeedbackDlg(display_text=display_text, logger_name='xms.rsm', worker=worker, parent=None)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    if feedback_dlg.exec():
        if LogEchoQSignalStream.logged_error:
            XmEnv.report_export_error()
    else:
        XmEnv.report_export_aborted()
