"""SimComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.gmi.component_bases.dataset_selector_mixin import DatasetSelectorMixin
from xms.gmi.component_bases.sim_component_base import SimComponentBase
from xms.gmi.data.generic_model import Section
from xms.guipy.dialogs.process_feedback_dlg import run_feedback_dialog

# 4. Local modules
from xms.schism.data.model import get_model
from xms.schism.feedback.map_coverage_runner import MapCoverageRunner
from xms.schism.feedback.map_tides_runner import MapTidesRunner
from xms.schism.feedback.map_upwind_solver_coverage_thread import MapUpwindSolverCoverageThread


class SimComponent(SimComponentBase, DatasetSelectorMixin):
    """A hidden Dynamic Model Interface (DMI) component for the SCHISM model simulation."""
    def _get_global_parameter_section(self) -> Section:
        """Get the global parameter section."""
        return get_model().global_parameters

    def item_linked(self, query: Query, linked_uuid: str, unique_name: str, lock_state: bool, parent: QWidget):
        """
        Handle when a new item was linked to the simulation.

        Args:
            query: Interprocess communication object.
            linked_uuid: UUID of the item that was just linked.
            unique_name: The unique-name of the item being linked.
            lock_state: Whether the item is currently locked for editing. Currently only makes sense in GMS.
            parent: Parent widget.
        """
        if unique_name == 'domain':
            return [], []
        elif unique_name == 'boundary_conditions':
            worker = MapCoverageRunner(coverage_uuid=linked_uuid, query=query)
        elif unique_name == 'tidal_constituents':
            worker = MapTidesRunner(tide_uuid=linked_uuid, query=query)
        elif unique_name == 'UpwindSolverCoverageComponent':
            worker = MapUpwindSolverCoverageThread(coverage_uuid=linked_uuid, query=query)
        else:
            raise AssertionError('Unknown take')
        run_feedback_dialog(worker, parent)
        return [], []
