"""Module for the Timer class."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
import time

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules

FIVE_SECONDS = 5
S_TO_NS = 10**9


class Timer:
    """
    A timer that can be used to log output at regular intervals.

    This is useful for long-running tasks, where you'd like to give some output to reassure the user that the process
    hasn't hung, but don't want to spam the log with millions of lines of output.

    The intended use case is that you construct the timer with the interval you'd like to report at, then look at
    `self.should_report` to decide whether to report. For example,

    >>> timer = Timer()
    ... lines = open('input.txt').readlines()
    ... for i in range(len(lines)):
    ...   line = lines[i]
    ...   # do something with line
    ...   if timer.should_report:
    ...     print(f'{i + 1} / {len(lines)} lines processed')  # only runs once every 5 seconds

    Note that reports are not queued, so if you ask for a five-second interval, then check at 6 seconds and 10 seconds,
    the first check will be `True` since at least five seconds passed, but the second will be `False` since less than
    five seconds passed since the last report.
    """
    def __init__(self, interval: int = FIVE_SECONDS):
        """
        Initialize the timer.

        Args:
            interval: How long the timer should wait, in nanoseconds, before recommending reporting again.
        """
        self.last_report = time.monotonic_ns()
        self.interval = interval * S_TO_NS

    @property
    def should_report(self):
        """
        Check whether another report should be made now.

        This returns `True` approximately once every `interval` nanoseconds.
        """
        if time.monotonic_ns() - self.last_report >= self.interval:
            self.last_report = time.monotonic_ns()
            return True
        return False
