"""Module for the DisplayOptionsHelper class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['DisplayOptionsHelper', 'load_component_id_map']

# 1. Standard Python modules
from functools import cached_property
from itertools import cycle
import os
from pathlib import Path
from typing import Optional, Sequence

# 2. Third party modules
from PySide2.QtGui import QColor

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.components.bases.coverage_component_base import CoverageComponentBase
from xms.components.display.display_options_io import (
    read_display_options_from_json, write_display_option_ids, write_display_option_line_locations,
    write_display_option_polygon_locations, write_display_options_to_json
)
from xms.grid.ugrid import UGrid
from xms.guipy.data.category_display_option import CategoryDisplayOption
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.line_style import LineOptions
from xms.guipy.data.point_symbol import PointOptions
from xms.guipy.data.polygon_texture import PolygonOptions
from xms.guipy.data.target_type import TargetType

# 4. Local modules

DEFAULT_COLORS = [
    (255, 0, 0),  # red
    (0, 0, 255),  # blue
    (85, 170, 0),  # lighter green
    (255, 204, 0),  # yellow
    (0, 160, 203),  # cyan
    (255, 0, 204),  # magenta
    (153, 0, 255),  # purple
    (255, 102, 0),  # orange
    (51, 153, 51),  # dark green
    (153, 102, 0),  # brown
    (102, 153, 153),  # grey blue
    (153, 51, 51),  # dark red
    (255, 153, 255),  # light magenta
]

INITIAL_POINT_ATT_ID_FILE = 'initial_point.attids'
INITIAL_POINT_COMP_ID_FILE = 'initial_point.compids'
INITIAL_ARC_ATT_ID_FILE = 'initial_arc.attids'
INITIAL_ARC_COMP_ID_FILE = 'initial_arc.compids'
INITIAL_POLYGON_ATT_ID_FILE = 'initial_polygon.attids'
INITIAL_POLYGON_COMP_ID_FILE = 'initial_polygon.compids'


class DisplayOptionsHelper:
    """Class for setting up display options for a component."""
    def __init__(self, main_file: str | Path):
        """
        Initialize the helper.

        Args:
            main_file: Mainfile of the component to set up display options for.
        """
        self._main_file = Path(main_file)
        self._dir = self._main_file.parent
        self._uuid = self._main_file.parent.name
        self._feature_type_to_index = {}

    @cached_property
    def display_directory(self) -> Path:
        """
        The directory containing the display options.

        This directory is guaranteed to exist.
        """
        self._dir.mkdir(parents=True, exist_ok=True)
        return self._dir

    def add_feature_types(self, target: TargetType, types: Sequence[str]):
        """
        Add some feature types to the display options.

        Args:
            target: Target type to add types for.
            types: The types to add.
        """
        if len(types) < 1:
            return None

        name = _target_to_name(target)
        pool = cycle(DEFAULT_COLORS)

        cats = CategoryDisplayOptionList()
        cats.target_type = target
        cats.comp_uuid = self._uuid

        for idx, t in enumerate(types):
            cats.categories.append(CategoryDisplayOption())
            c = cats.categories[-1]
            c.options = _target_to_options(target)
            c.description = t
            c.id = idx
            c.file = f'{name}_{idx}.ids'
            clr = next(pool)
            c.options.color = QColor(clr[0], clr[1], clr[2], 255)
            self._feature_type_to_index[(name, t)] = idx

        file_name = self._json_for_type(target)
        write_display_options_to_json(file_name, cats)

    def add_feature_ids(self, target: TargetType, feature_ids: Sequence[int]):
        """
        Add the IDs of some features in this coverage.

        Assumes there are no existing assigned component IDs, so feature IDs can be used as component IDs too.

        Not applicable to mapped coverages.

        Args:
            target: point, arc, or polygon.
            feature_ids: IDs to add.
        """
        name = _target_to_name(target)
        feature_file = self.display_directory / f'initial_{name}.attids'
        component_file = self.display_directory / f'initial_{name}.compids'
        write_display_option_ids(feature_file, feature_ids)
        write_display_option_ids(component_file, feature_ids)

    def map_features_to_type(self, target: TargetType, feature_ids: Sequence[int], feature_type: str):
        """
        Make features render as a particular type.

        The feature will be rendered with the color and label assigned to the given type.

        Not applicable to mapped coverages.

        Args:
            target: point, arc, or polygon.
            feature_ids: IDs of features to map.
            feature_type: The type to map to.
        """
        # Technically this should map component IDs, but this is first time initialization, so we can get
        # away with assuming each feature's feature ID is also its component ID and provide a nicer API.
        name = _target_to_name(target)
        idx = self._feature_type_to_index[(name, feature_type)]
        file_name = self.display_directory / f'{name}_{idx}.ids'
        write_display_option_ids(file_name, feature_ids)

    def draw_lines(self, feature_type: str, lines: Sequence[Sequence[tuple[float, float, float]]]):
        """
        Draw one or more lines.

        Only applicable to mapped coverages.

        Args:
            feature_type: The feature type to apply to the lines. Must have been previously passed to
                `self.add_feature_types(TargetType.arc, ...)`.
            lines: Sequence of lines to draw. Each line is a sequence of points. Each point is a tuple of x, y, z.
        """
        self._convert_to_drawing()

        flattened_lines = []
        for line in lines:
            flattened_line = []
            for location in line:
                flattened_line.extend(location)
            flattened_lines.append(flattened_line)

        name = _target_to_name(TargetType.arc)
        index = self._feature_type_to_index[(name, feature_type)]
        file_name = self.display_directory / f'{name}_{index}.locations'

        write_display_option_line_locations(file_name, flattened_lines)

    def draw_ugrid(self, ugrid: UGrid, cell_to_feature_index: Sequence[int], default_type: Optional[int] = None):
        """
        Draw a ugrid on the component.

        This effectively draws the UGrid, with each cell painted a particular color and texture. A cell's paint is
        determined by using its index in the grid to look up a feature index in `cell_to_feature_index`, then using that
        index to look up the cell's feature name in the list passed to
        `self.add_feature_types(TargetType.polygon, ...)`.

        If `default_type` is provided, cells with feature index `-1` are assigned the default type. This use is typical
        when a coverage's polygons have been snapped to the UGrid using `xms.snap.snap_polygon.SnapPolygon`. If the
        NumPy array retrieved from `SnapPolygon` is named `mapping`, then you can use `mapping - 1` to shift everything
        left by one and get a suitable `cell_to_feature_index` for this method.

        When reading from model-native files, it is more typical that every cell has already been identified, so the
        use of a default type is usually unnecessary.

        See `xms.snap.snap_polygon.SnapPolygon` for a tool that can be used to get from the polygons in a coverage to
        the mapping used here.

        Args:
            ugrid: The geometry to draw.
            cell_to_feature_index: A mapping from cell index in the UGrid to an index in the list passed to
                `self.add_feature_types(TargetType.polygon, ...)`.
            default_type: Type to assign to any cells that `cell_to_feature_index` maps to -1.
        """
        self._convert_to_drawing()

        num_types = self._num_feature_types(TargetType.polygon)
        polygon_lists = _build_polygon_lists(ugrid, cell_to_feature_index, default_type, num_types)

        name = _target_to_name(TargetType.polygon)
        for index, polygon_list in enumerate(polygon_lists):
            path = self.display_directory / f'{name}_{index}.locations'
            write_display_option_polygon_locations(path, polygon_list)

    def _json_for_type(self, target_type: TargetType) -> Path:
        """
        Get the path to the .json file for a target type.

        Args:
            target_type: Target to get the .json file for.

        Returns:
            Path to the .json file.
        """
        name = _target_to_name(target_type)
        path = self.display_directory / f'{name}_display.json'
        return path

    def _file_for_type(self, target_type: TargetType, feature_type: str) -> str:
        """
        Get the ID/location file for a particular feature type.

        Args:
            target_type: The feature's target type.
            feature_type: The desired target type.

        Returns:
            The name of the file where IDs/locations for this feature type should be written to.
        """
        name = _target_to_name(target_type)
        file_name = self.display_directory / f'{name}_display.json'
        json = read_display_options_from_json(file_name)
        categories = CategoryDisplayOptionList()
        categories.from_dict(json)
        for category in categories.categories:
            if category.description == feature_type:
                return category.file

    def _convert_to_drawing(self):
        """Convert the display options to drawing mode."""
        for target in [TargetType.point, TargetType.arc, TargetType.polygon]:
            name = _target_to_name(target)
            file_name = self.display_directory / f'{name}_display.json'
            if file_name.exists():
                json = read_display_options_from_json(file_name)
                json['is_ids'] = 0
                categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
                categories.from_dict(json)
                for i in range(len(categories.categories)):
                    original_name = categories.categories[i].file
                    new_name = original_name.replace('.ids', '.locations')
                    categories.categories[i].file = new_name
                write_display_options_to_json(file_name, categories)

    def _num_feature_types(self, target: TargetType) -> int:
        """
        Get the number of feature types for a given target type.

        Args:
            target: The target type to get the number of features for.

        Returns:
            The number of feature types.
        """
        path = self._json_for_type(target)
        if not path.exists():
            return 0
        json = read_display_options_from_json(path)
        categories = CategoryDisplayOptionList()
        categories.from_dict(json)
        return len(categories.categories)


def load_component_id_map(
    query: Query, component: CoverageComponentBase, target_types: Optional[list[TargetType]] = None
):
    """
    Load a coverage's component ID map from XMS.

    Args:
        query: Interprocess communication object.
        component: Component to load map for.
        target_types: Desired target types to load. Points, arcs, and polygons are supported. If None, loads everything.
    """
    if target_types is None:
        target_types = [TargetType.point, TargetType.arc, TargetType.polygon]

    params = {
        'points': TargetType.point in target_types,
        'arcs': TargetType.arc in target_types,
        'polygons': TargetType.polygon in target_types
    }

    files_dict = query.load_component_ids(component, **params, delete_files=False)
    component.load_coverage_component_id_map(files_dict)

    for att_file, comp_file in files_dict.values():
        if os.path.isfile(att_file):
            os.remove(att_file)
        if os.path.isfile(comp_file):
            os.remove(comp_file)


def _target_to_name(target: TargetType) -> str:
    """
    Get a name for a target.

    Args:
        target: The target to get the name of.

    Returns:
        A string like 'point', 'arc', or 'poly'.
    """
    if target == TargetType.point:
        return 'point'
    elif target == TargetType.arc:
        return 'arc'
    elif target == TargetType.polygon:
        return 'poly'
    else:
        raise ValueError('Unsupported target')


def _target_to_options(target: TargetType) -> PointOptions | LineOptions | PolygonOptions:
    """
    Get either PointOptions, LineOptions, or PolygonOptions, depending on target type.

    Args:
        target: The target type to get options for.

    Returns:
        The *Options type that matches the provided target.
    """
    if target == TargetType.point:
        return PointOptions()
    elif target == TargetType.arc:
        return LineOptions()
    elif target == TargetType.polygon:
        return PolygonOptions()
    else:
        raise ValueError('Unsupported target')


def _build_polygon_lists(ugrid: UGrid, mapping: Sequence[int], default_type: int,
                         num_feature_types: int) -> list[list[dict[str, list[int]]]]:
    """
    Build lists of polygons.

    Args:
        ugrid: The geometry to build polygons for.
        mapping: Mapping from cell_index -> feature_index. Used to determine which display options to apply to a cell.
        default_type: The feature index to assign to cells with feature_index==-1.
        num_feature_types: How many feature types to expect.

    Returns:
        List of lists of polygons. Each element of the list is suitable for passing to
        `write_display_option_polygon_locations`.
    """
    polygon_lists = [[] for _ in range(num_feature_types)]
    for cell_index in range(ugrid.cell_count):
        cell_locations = ugrid.get_cell_locations(cell_index)
        stream = [component for location in cell_locations for component in location]
        if len(stream) < 9:  # <9 components == <3 points == degenerate cell
            continue  # pragma: no cover
        stream.extend(stream[:3])  # Repeat the first point to close the loop
        type_index = mapping[cell_index]
        if type_index == -1:
            type_index = default_type
        polygon = {'outer': stream}
        polygon_lists[type_index].append(polygon)
    return polygon_lists
