"""Module for exporting a SCHISM simulation with feedback."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"
__all__ = ['ExportSimulationRunner']

# 1. Standard Python modules
from functools import cached_property
from pathlib import Path

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.constraint import UGrid2d
from xms.core.filesystem.filesystem import copyfile
from xms.gmi.data.generic_model import Section
from xms.guipy.dialogs.feedback_thread import ExpectedError, FeedbackThread

# 4. Local modules
from xms.schism.data.mapped_bc_data import MappedBcData
from xms.schism.data.mapped_upwind_solver_coverage_data import MappedUpwindSolverCoverageData
from xms.schism.data.model import get_model, needed_files, parameter_for_file
from xms.schism.data.sim_data import SimData
from xms.schism.dmi.xms_data import XmsData
from xms.schism.external.crc import compute_crc
from xms.schism.external.mapped_tidal_data import MappedTidalData
from xms.schism.external.project import make_geographic
from xms.schism.file_io import (
    BcTidesFile, Boundary, Fort14File, VGridFile, write_bctides, write_fort14, write_namelist, write_vgrid
)


class ExportSimulationRunner(FeedbackThread):
    """Class for exporting a SCHISM simulation in a worker thread."""
    def __init__(self, query: Query):
        """Constructor."""
        super().__init__(is_export=True, create_query=False)
        self.data = XmsData(query)

        self.display_text = {
            'title': 'SCHISM Export Simulation',
            'working_prompt': 'Exporting SCHISM simulation files. Please wait...',
            'warning_prompt': 'Warning(s) encountered while exporting simulation. Review log output for more details.',
            'error_prompt': 'Error(s) encountered while exporting simulation. Review log output for more details.',
            'success_prompt': 'Successfully exported simulation',
            'note': '',
            'auto_load': 'Close this dialog automatically when exporting is finished.'
        }

        self._domain_hash = ''

    @cached_property
    def _sim_data(self) -> SimData:
        """The data manager for the simulation component."""
        self._log.info('Retrieving simulation data...')
        main_file = self.data.sim_main_file
        sim_data = SimData(main_file)
        return sim_data

    @cached_property
    def _section(self) -> Section:
        """The global parameters for the model."""
        data = self._sim_data
        global_parameters = get_model().global_parameters
        global_parameters.restore_values(data.global_values)
        return global_parameters

    @cached_property
    def _domain(self) -> UGrid2d:
        """The model's mesh."""
        self._log.info('Retrieving domain...')
        domain = self.data.ugrid
        if domain is None:
            raise ExpectedError('No domain was linked to the simulation.')
        self._domain_hash = compute_crc(self.data.ugrid_file)
        return domain

    @cached_property
    def _mapped_component_data(self) -> MappedBcData:
        """The mapped boundary condition coverage."""
        self._log.info('Retrieving applied boundary conditions...')
        main_file = self.data.mapped_boundary_coverage_file
        if main_file is None:
            raise ExpectedError('No mapped coverage was linked to the simulation.')

        data = MappedBcData(main_file)
        if data.domain_hash != self._domain_hash:
            raise ExpectedError(
                'The domain was changed since the coverage was mapped. '
                'The boundary condition coverage must be remapped before exporting. '
                'If tides are mapped, they may also need to be re-mapped.'
            )
        return data

    @cached_property
    def _mapped_tides(self) -> MappedTidalData:
        """The mapped tidal simulation."""
        self._log.info('Retrieving tidal information...')

        main_file = self.data.mapped_tides_main_file
        if main_file is None:
            raise ExpectedError('No mapped tides were linked to the simulation.')

        data = MappedTidalData(main_file)
        if data.domain_hash != self._domain_hash:
            raise ExpectedError(
                'The domain was changed since the tides were mapped. '
                'Tides must be re-mapped before exporting the simulation.'
            )
        return data

    @cached_property
    def _mapped_upwind_data(self) -> MappedUpwindSolverCoverageData:
        """The mapped upwind solver coverage."""
        self._log.info('Retrieving upwind solver data...')

        main_file = self.data.mapped_solver_main_file
        if main_file is None:
            raise ExpectedError('No upwind solver coverage was linked to the simulation.')

        data = MappedUpwindSolverCoverageData(main_file)
        if data.domain_hash != self._domain_hash:
            raise ExpectedError(
                'The domain was changed since the upwind solver coverage was mapped. '
                'The upwind solver coverage must be re-mapped before exporting the simulation.'
            )
        return data

    def _write_hotstart(self) -> None:
        """Copy a hotstart file."""
        hotstart_source = Path(self._section.group('opt').parameter('ihot_file').value)
        hotstart_dest = Path('./hotstart.nc')

        if hotstart_source.is_file():
            self._log.info('Copying hotstart.nc...')
            copyfile(hotstart_source, hotstart_dest)
        elif hotstart_dest.is_file():
            self._log.info(f'`{hotstart_source}` did not exist. Used hotstart.nc from previous export.')
        else:
            raise ExpectedError(
                f'Unable to copy hotstart.nc from `{hotstart_source}` and no hotstart.nc found in simulation folder. '
                'If this is the wrong path, it can be changed by setting the ihot_file parameter in the opt '
                'section of the model control dialog.'
            )

    def _write_param_nml(self):
        """Write param.nml."""
        section = self._section
        sim_data = self._sim_data
        section.restore_values(sim_data.global_values)
        write_namelist(section, Path('param.nml'))

    def _write_hgrid_gr3(self):
        """Write hgrid.gr3."""
        domain = self._domain
        data = self._mapped_component_data
        elevations = np.array(domain.point_elevations)
        depths = -elevations
        self._log.info('Writing domain...')

        open_arcs = [Boundary(boundary_type=0, nodes=nodes) for nodes, _ in data.open_arcs]
        closed_arcs = [Boundary(boundary_type=flag, nodes=nodes) for nodes, flag in data.closed_arcs]

        fort14 = Fort14File(
            ugrid=domain,
            dataset=depths,
            open_boundaries=open_arcs,
            closed_boundaries=closed_arcs,
        )

        write_fort14(fort14, 'hgrid.gr3')

    def _write_bctides_in(self):
        """Write bctides.in."""
        self._log.info('Writing boundary conditions...')

        cutoff_depth = self._section.group('other').parameter('tip_dp').value

        data = self._mapped_component_data
        boundaries = [nodes for nodes, _ in data.open_arcs]
        values = [values for _, values in data.open_arcs]

        tides = self._mapped_tides

        bctides = BcTidesFile(
            cutoff_depth=cutoff_depth,
            open_boundaries=boundaries,
            values=values,
            forcing_frequencies=tides.properties,
            elevation=tides.elevation,
            velocity=tides.velocity,
        )
        write_bctides(bctides, 'bctides.in')

    def _write_vgrid_in(self):
        """Write vgrid.in."""
        self._log.info('Writing vgrid...')
        group = self._section.group('other')

        z_levels = group.parameter('z_levels').value
        if not z_levels:
            raise ExpectedError('No Z-grid levels defined.')
        z_levels = [z[0] for z in z_levels]  # The value is a list of rows, we need a flat list.

        s_levels = group.parameter('s_levels').value
        if not s_levels:
            raise ExpectedError('No S-grid levels defined.')
        s_levels = [s[0] for s in s_levels]
        if s_levels[0] != 1.0 or s_levels[-1] != 0.0:
            raise ExpectedError('S-grid levels must start at 1.0 and end at 0.0')

        vgrid = VGridFile(
            hc=group.parameter('hc').value,
            theta_b=group.parameter('theta_b').value,
            theta_f=group.parameter('theta_f').value,
            z_levels=z_levels,
            s_levels=s_levels
        )

        write_vgrid(vgrid, 'vgrid.in')

    def _write_hgrid_ll(self):
        """Write hgrid.ll if necessary."""
        success, ugrid = make_geographic(self.data.display_projection, self._domain)
        if success:
            file = Fort14File(ugrid=ugrid, dataset=ugrid.ugrid.locations[:, 2])
            write_fort14(file, 'hgrid.ll')

    def _write_sflux(self, pattern: str):
        """Write sflux/? if necessary."""
        sflux = Path('.')
        have_sflux = False
        for _ in sflux.glob(pattern):
            have_sflux = True
            break

        if not have_sflux:
            self._log.warning(
                f'{pattern} must be manually saved to the model model directory. '
                'See https://schism-dev.github.io/schism/master/input-output/sflux.html '
                'for instructions on obtaining the required files.'
            )

    def _write_tvd_prop(self):
        """Write the tvd.prop file."""
        self._log.info('Writing tvd.prop...')

        mapped_upwind = self._mapped_upwind_data
        solver_map = mapped_upwind.solver

        with open('tvd.prop', 'w') as f:
            for i, flag in enumerate(solver_map, start=1):
                f.write(f'{i} {flag}\n')

    def _write_gr3(self, file_name: str):
        """
        Write a .gr3 file.

        Most .gr3 files are actually dataset files. This method only writes dataset files.
        Some files, such as hgrid.gr3 and hgrid.ll, also include geometry. This method does
        not write such files.

        This function also handles .ic files, which are actually just .gr3 files with a
        different extension.

        Args:
            file_name: Where to write the dataset to.
        """
        group_name, parameter_name = parameter_for_file(file_name)

        parameter = self._section.group(group_name).parameter(parameter_name)
        uuid = parameter.value
        if not uuid:
            raise ExpectedError(f"No dataset specified for parameter '{parameter.label}'")

        data = self.data.datasets[uuid]
        if data is None:
            raise ExpectedError(f"Parameter '{parameter.label}' refers to a dataset that no longer exists.")

        self._log.info(f'Writing {file_name}...')
        fort14 = Fort14File(ugrid=self._domain, dataset=data)
        write_fort14(fort14, file_name)

    def _write_optional_files(self):
        """Write all the optional files."""
        try:
            files = needed_files(self._section)
        except ValueError as err:
            raise ExpectedError(err.args[0])

        for file in sorted(files):  # For test stability. Order affects recording test log.
            self._write_optional_file(file)

    def _write_optional_file(self, file: str):
        """
        Write an optional file.

        Args:
            file: Name of the file to write.
        """
        if file.endswith('.gr3') or file.endswith('.ic'):
            self._write_gr3(file)
        elif file in ['sflux/sflux_air_1.*.nc', 'sflux/sflux_prc_1.*.nc', 'sflux/sflux_rad_1.*.nc']:
            self._write_sflux(file)
        elif file == 'tvd.prop':
            self._write_tvd_prop()
        elif file == 'hotstart.nc':
            self._write_hotstart()
        else:  # pragma: nocover
            raise ExpectedError(f'Unsupported file: {file}')

    def _run(self):
        """Export a simulation."""
        self._write_param_nml()
        self._write_hgrid_gr3()
        self._write_bctides_in()
        self._write_hgrid_ll()
        self._write_vgrid_in()

        self._write_optional_files()
