"""Reader for fort14 (.gr3) files."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
from array import array
from pathlib import Path
from typing import Sequence
import uuid

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules
from xms.constraint import UGridBuilder
from xms.grid.ugrid import UGrid

# 4. Local modules
from .fort14_file import Boundary, Fort14File
from ..reader_base import ReaderBase


def read_fort14(path: Path | str, read_boundaries: bool = False) -> Fort14File:
    """
    Read a fort14, AKA .gr3, file.

    Args:
        path: The file to read.
        read_boundaries: Whether to read boundaries from the file. The hgrid.gr3 file should have boundaries, while
            all the rest are really just an overly complex way to represent a dataset.

    Returns:
        The read file.
    """
    with Fort14Reader(path) as reader:
        reader.read_domain()
        xmugrid = reader.ugrid

        builder = UGridBuilder()
        builder.set_is_2d()
        builder.set_ugrid(xmugrid)

        if read_boundaries:
            reader.read_boundaries()

    ugrid = builder.build_grid()
    ugrid.uuid = str(uuid.uuid4())
    return Fort14File(
        ugrid=ugrid,
        dataset=reader.z,
        open_boundaries=reader.open_boundaries,
        closed_boundaries=reader.closed_boundaries
    )


class Fort14Reader(ReaderBase):
    """Reader for fort14 (.gr3) files."""
    def __init__(self, path: Path | str):
        """
        Initialize the reader.

        Args:
            path: Path to the file to read from.
        """
        super().__init__(path, ['!', '='])

        self._node_id_to_index = {}

        self.ugrid = UGrid()
        self.z: Sequence[float] = []
        self.open_boundaries: list[Boundary] = []
        self.closed_boundaries: list[Boundary] = []

    def read_domain(self):
        """
        Read the geometry of the file.

        The geometry includes locations, elevations, and connectivity.
        """
        self._read_line()  # The first line is just a comment.

        num_elements, num_nodes = self._parse_line(int, int)

        self._parse_nodes(num_nodes)
        self._parse_elements(num_elements)

    def _parse_nodes(self, num_nodes: int):
        """
        Read all the nodes out of the file.

        Args:
            num_nodes: Number of nodes to expect.
        """
        locations = np.full((num_nodes, 3), 0.0, dtype=float)
        zs = np.full((num_nodes, ), 0.0, dtype=float)
        mapping = {}

        for index in range(num_nodes):
            node_id, x, y, z = self._parse_line(int, float, float, float)
            locations[index] = (x, y, z)
            zs[index] = z
            mapping[int(node_id)] = index

        self.ugrid.locations = locations
        self.z = zs
        self._node_id_to_index = mapping

    def _parse_elements(self, num_elements):
        """
        Read the connectivity out of the file.

        Args:
            num_elements: Number of elements to expect.
        """
        cell_stream = array('l')
        try:
            for _ in range(num_elements):
                nums = [int(num) for num in self._read_line().split()]
                node_id, length, *node_ids = nums
                if len(node_ids) != length:
                    raise self._error('Wrong number of fields.')
                if not (3 <= length <= 4):
                    raise self._error('Only triangles and quads supported.')
                cell_type = UGrid.cell_type_enum.QUAD if length == 4 else UGrid.cell_type_enum.TRIANGLE
                node_indexes = [self._node_id_to_index[node_id] for node_id in node_ids]
                cell_stream.extend((cell_type, length, *node_indexes))
        except ValueError:
            raise self._error('Must be 5 or 6 integers.')
        except KeyError as exc:
            raise self._error(f'Reference to undefined node {exc}.')

        self.ugrid.cellstream = cell_stream

    def read_boundaries(self):
        """Read the boundary definitions out of the file."""
        num_open_boundaries = self._parse_line(int)
        self._parse_boundary_set(num_open_boundaries, self.open_boundaries)
        num_closed_boundaries = self._parse_line(int)
        self._parse_boundary_set(num_closed_boundaries, self.closed_boundaries)

    def _parse_boundary_set(self, num_boundaries: int, boundaries: list[Boundary]):
        """
        Parse a set of boundaries out of the file.

        This handles both the open and closed boundary sets.

        Args:
            num_boundaries: Number of boundaries to parse.
            boundaries: A list to append boundaries on to. Should probably be empty.
        """
        self._read_line()  # Number of nodes in set. SCHISM probably uses it for pre-allocation; we don't need it.
        for _ in range(num_boundaries):
            self._parse_boundary(boundaries)

    def _parse_boundary(self, boundaries: list[Boundary]):
        """
        Parse a single boundary out of the file.

        Args:
            boundaries: A list to append boundaries on to.
        """
        try:
            nodes = array('l')
            num_nodes, *flag = self._read_line(comment_markers='!=').split()
            num_nodes = int(num_nodes)
            if flag:
                flag = int(flag[0])
            else:
                flag = 0
            for _ in range(num_nodes):
                nodes.append(int(self._read_line()))
            boundary = Boundary(boundary_type=flag, nodes=nodes)
            boundaries.append(boundary)
        except ValueError:
            raise self._error('Expected 1 or 2 integers.')
