"""CheckSimulation class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as xmf
from xms.grid.geometry import geometry as gm

# 4. Local modules


class CheckSimulation:
    """Model check for SRH-2D simulation."""
    def __init__(self, check_runner):
        """Constructor."""
        super().__init__()
        self.error_text = ''
        self._model_control = None
        self._sim_comp_dir = ''
        if check_runner.sim_component:
            self._model_control = check_runner.sim_component.data
            self._sim_comp_dir = os.path.dirname(check_runner.sim_component.main_file)
        self._ugrid = check_runner.ugrid
        self._co_grid = check_runner.co_grid
        self._grid_units = check_runner.grid_units
        self._bc_data = check_runner.bc_data
        self._bc_comp_id_to_arc_id = check_runner.bc_comp_ids_to_arc_ids
        self._wse_dataset = check_runner.wse_dataset
        self._mat_data = check_runner.mat_data
        self._nlcd_raster = check_runner.nlcd_raster
        self.check_cell_areas = True
        self._log = logging.getLogger('xms.srh')

    def run_check(self):
        """Runs model check on the simulation."""
        self._check_mesh()
        self._check_bcs()
        self._check_materials()
        self._check_output()
        self._check_case_name()
        self._check_initial_conditions()
        self._check_time_step()

    def _check_mesh(self):
        """Check metrics on the mesh."""
        # make sure a mesh is in the simulation
        if not self._ugrid:
            self._log.error('SRH2D requires a mesh to run. Aborting. Add a mesh to the simulation to remove this error')
            return

        if self._co_grid:
            if not self._co_grid.check_all_cells_2d():
                self._log.error('Non-2d cells detected. SRH2D requires that all cells be 2D.')
                raise RuntimeError

        # check the number of elements in the mesh
        num_cells = self._ugrid.cell_count
        if num_cells < 200000:
            pass
        else:
            msg = f'The unstructured mesh contains {num_cells} elements.'
            if num_cells < 500000:
                msg += ' Best performance for SRH-2D occurs with meshes with under 100,000 elements.'
                self._log.warning(msg)
            elif num_cells < 2000000:
                msg += ' Poor performance for SRH-2D will occur with this mesh.'
                self._log.warning(msg)
            else:
                msg += ' The existing mesh greatly exceeds the maximum number of elements.'
                self._log.error(msg)

        # check the units for the mesh (must be feet or meters)
        if not self._grid_units:
            msg = 'Horizontal units are not FEET or METERS.'
            self._log.error(msg)

        if not self.check_cell_areas:
            return

        small_area_cells = []
        locs = self._ugrid.locations
        for idx in range(num_cells):
            pts = [locs[p] for p in self._ugrid.get_cell_points(idx)]
            if gm.polygon_area_2d(pts) < 0.1:
                small_area_cells.append(idx + 1)

        if small_area_cells:
            msg = 'The unstructured mesh contains the following elements with an area less than 0.1.'
            self._log.warning(msg)
            self._log.warning(f'Element ids: {small_area_cells}')

    def _check_bcs(self):
        """Check the boundary conditions."""
        # bc coverage must exist in the simulation
        if not self._bc_data:
            msg = 'STOP! A boundary condition coverage must be included in simulation.'
            self._log.error(msg)

    def _check_materials(self):
        """Check the material properties."""
        # material coverage must exist in the simulation
        if not self._mat_data:
            msg = 'STOP! A material coverage must be included in simulation.'
            if not self._nlcd_raster:
                self._log.error(msg)
            return

        # must have material defined in addition to 'unassigned'
        df = self._mat_data.materials.to_dataframe()
        mat_names = df['Name'].to_list()
        if len(mat_names) < 2:
            msg = 'STOP! No user defined material zones found. Add materials to the material coverage.'
            self._log.error(msg)

        # material names must be unique
        unique_names = set(mat_names)
        if len(mat_names) != len(unique_names):
            msg = 'STOP! Material names must be unique. Edit the material names to in the material ' \
                  'coverage to be unique.'
            self._log.error(msg)

    def _check_output(self):
        """Checks the simulation outputs."""
        # check that output units are consistent with grid units
        out_units = 'GridUnit "FOOT"' if self._model_control.output.output_units == 'English' else 'GridUnit "METER"'
        if out_units != self._grid_units:
            msg = 'Warning: Output units inconsistent with grid units. Edit the output units in' \
                  ' the simulation model control.'
            self._log.warning(msg)

        # output format should be XMDF
        if self._model_control.output.output_format != 'XMDF':
            msg = 'Warning: Selected output type is not readable by SMS. XMDF is the format supported by SMS' \
                  ' which SRH-2D outputs.Edit the output format in the simulation model control.'
            self._log.warning(msg)

    def _check_case_name(self):
        """Checks the case name."""
        case_name = self._model_control.hydro.case_name
        # check for empty case name
        if not case_name:
            msg = 'No case name has been specified. SRH-2D requires a valid case name to build ' \
                  'output file names.'
            self._log.error(msg)

        # check for illegal characters in the case name
        illegals = r'\/:*?"<>|'
        if any(illegal in case_name for illegal in illegals):
            fix = r'Edit the case name in the simulation model control. The case name may not contain' \
                  r' any of the following characters \ / : * ? " < > |.'  # noqa
            msg = f'"{case_name}" is not a valid case name. ' + fix
            self._log.error(msg)

        # check length of case name must be < 39 characters
        if len(case_name) > 38:
            msg = f'"{case_name}" is longer than 38 characters. Edit the case name so that the ' \
                  f'length is less than 39 characters.'
            self._log.error(msg)

    def _check_initial_conditions(self):
        """Check simulation initial conditions."""
        if self._model_control.hydro.initial_condition == 'Restart File':
            # make sure that the restart file exists
            restart_file = ''
            if self._model_control.hydro.restart_file:
                restart_file = xmf.resolve_relative_path(self._sim_comp_dir, self._model_control.hydro.restart_file)
            if not restart_file or not os.path.isfile(restart_file):
                msg = 'No initial condition restart file has been specified. Specify a restart file in' \
                      ' the simulation model control.'
                self._log.error(msg)

        if self._model_control.hydro.initial_condition == 'Water Surface Elevation Dataset':
            if self._wse_dataset is None:
                msg = 'No initial condition water surface elevation dataset specified. Specify a ' \
                      'water surface elevation dataset in the simulation model control.'
                self._log.error(msg)

    def _check_time_step(self):
        """Check simulation timestep."""
        if self._model_control.hydro.time_step <= 0.0:
            msg = 'Time step must be greater than 0.0. Edit the time step in the model control dialog.'
            self._log.error(msg)
