"""BcComponent class. Data for Bc Coverage."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import filecmp
import json
import os

# 2. Third party modules
import pkg_resources
from PySide2.QtGui import QIcon

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.components.display.display_options_io import (
    read_display_option_ids, read_display_options_from_json, write_display_option_ids, write_display_options_to_json
)
from xms.components.display.xms_display_message import XmsDisplayMessage
from xms.core.filesystem import filesystem
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.plot_and_table_data_srh import PlotsAndTableDataSrh
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog
from xms.guipy.dialogs.plot_and_table_dialog import PlotsAndTableDialog
from xms.guipy.dialogs.xms_parent_dlg import get_xms_icon

# 4. Local modules
from xms.srh.components.bc_data_dialog_runner import BcDataDialogRunner
from xms.srh.components.srh_component import duplicate_display_opts
from xms.srh.components.srh_cov_component import SrhCoverageComponent
from xms.srh.data.bc_data import BcData
from xms.srh.file_io.hy8_migrate import Hy8Migrate

BC_INITIAL_ATT_ID_FILE = 'initial_bc.attids'
BC_INITIAL_COMP_ID_FILE = 'initial_bc.compids'


class BcComponent(SrhCoverageComponent):
    """A hidden Dynamic Model Interface (DMI) component for the SRH-2D model simulation."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.

        """
        super().__init__(main_file)
        # [(menu_text, menu_method)...]
        self.tree_commands = [
            ('Display Options...', 'open_display_options'),
        ]
        self.arc_commands = [
            ('Assign BC...', 'open_assign_arc_bc'),
            ('Structure Output Plots...', 'open_arc_structure_plot'),
        ]
        shared_dir = XmEnv.xms_environ_shared_directory()
        self.hy8_file = os.path.join(shared_dir, 'culvert.hy8')
        if not os.path.isfile(self.hy8_file):
            with open(self.hy8_file, 'w'):
                pass
        self.data = BcData(self.main_file)
        self.data.load_all()
        self.cov_uuid = self.data.info.attrs['cov_uuid']
        comp_dir = os.path.dirname(self.main_file)
        self.disp_opts_file = os.path.join(comp_dir, 'bc_display_options.json')
        if not os.path.exists(self.disp_opts_file):
            # Read the default display options, and save ourselves a copy with a randomized UUID.
            categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
            default_file = os.path.join(
                os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data',
                'default_bc_display_options.json'
            )
            json_dict = read_display_options_from_json(default_file)
            json_dict['comp_uuid'] = os.path.basename(os.path.dirname(self.main_file))
            categories.from_dict(json_dict)
            write_display_options_to_json(self.disp_opts_file, categories)
            # Save our display list UUID to the main file
            self.data.info.attrs['display_uuid'] = categories.uuid
            self.data.commit()

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (:obj:`str`): Path to the new save location.
            save_type (:obj:`str`): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:

                new_main_file (:obj:`str`): Name of the new main file relative to new_path, or an absolute path
                if necessary.

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)

        file_version = self.data.info.attrs['VERSION']
        tmp_dir = XmEnv.xms_environ_temp_directory()
        if pkg_resources.parse_version(file_version) < pkg_resources.parse_version('2.0.0'):
            if os.environ.get('XMS_OPENING_PROJECT', '') == 'TRUE':
                hy8_file = os.path.join(tmp_dir, 'hy8.txt')
                if not os.path.isfile(hy8_file):
                    with open(hy8_file, 'w') as _:
                        pass
                    action_requests.append(
                        ActionRequest(
                            modality='NO_DIALOG',
                            class_name=self.class_name,
                            module_name=self.module_name,
                            main_file=self.main_file,
                            method_name='upgrade_hy8_from_version_1'
                        )
                    )
        if os.environ.get('XMS_OPENING_PROJECT', '') == 'TRUE':
            hy8_file = os.path.join(tmp_dir, 'check_external_hy8.txt')
            if not os.path.isfile(hy8_file):
                with open(hy8_file, 'w') as _:
                    pass
                action_requests.append(
                    ActionRequest(
                        modality='NO_DIALOG',
                        class_name=self.class_name,
                        module_name=self.module_name,
                        main_file=self.main_file,
                        method_name='check_external_hy8'
                    )
                )

        if save_type == 'DUPLICATE':
            json_dict = duplicate_display_opts(new_path, os.path.basename(self.disp_opts_file))
            data = BcData(new_main_file)
            data.load_all()
            data.info.attrs['cov_uuid'] = ''
            data.info.attrs['display_uuid'] = json_dict['uuid']
            data.commit()

        return new_main_file, messages, action_requests

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (:obj:`bool`): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform..

        """
        self.data.commit()
        id_dir = os.path.join(os.path.dirname(self.main_file), 'display_ids')
        os.mkdir(id_dir)

        action = ActionRequest(
            main_file=self.main_file,
            modality='NO_DIALOG',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name='get_initial_display_options'
        )
        return [], [action]

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists

        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid
        self.do_init_disp_opts()
        return [], []

    def do_init_disp_opts(self):
        """Intial display options work."""
        initial_att_file = os.path.join(os.path.dirname(self.main_file), BC_INITIAL_ATT_ID_FILE)
        if os.path.isfile(initial_att_file):  # Came from a model native read, initialize the component ids.
            att_ids = read_display_option_ids(initial_att_file)
            initial_comp_file = os.path.join(os.path.dirname(self.main_file), BC_INITIAL_COMP_ID_FILE)
            comp_ids = read_display_option_ids(initial_comp_file)
            filesystem.removefile(initial_att_file)
            filesystem.removefile(initial_comp_file)
            for att_id, comp_id in zip(att_ids, comp_ids):
                self.update_component_id(TargetType.arc, att_id, comp_id)
        # Send the default material list to XMS.
        self.display_option_list.append(XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid))

    def open_assign_arc_bc(self, query, params, win_cont):
        """Opens the Assign Obstruction dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform..

        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid
        arc_ids = params[0].get('selection', [])
        num_arcs = len(arc_ids)
        if num_arcs == 0:
            return [('INFO', 'No arcs selected. Select one or more arcs to assign properties.')], []

        runner = BcDataDialogRunner(
            bc_component=self, sel_arc_ids=arc_ids, win_cont=win_cont, query=query, cov_uuid=self.cov_uuid
        )
        runner.run_dlg()
        return [], []

    def open_display_options(self, query, params, win_cont):
        """Shows the display options dialog.

        Args:
            query (:obj:`xms.api.dmi.Query`):
            params (:obj:`list[str]`):
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform..
        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid
        categories = CategoryDisplayOptionList()
        json_dict = read_display_options_from_json(self.disp_opts_file)
        categories.from_dict(json_dict)
        categories_list = [categories]

        dlg = CategoryDisplayOptionsDialog(categories_list, win_cont)
        icon_path = get_xms_icon()
        if icon_path:
            dlg.setWindowIcon(QIcon(icon_path))
        dlg.setModal(True)
        if dlg.exec():
            # write files
            category_lists = dlg.get_category_lists()
            for category_list in category_lists:  # pragma: no branch
                write_display_options_to_json(self.disp_opts_file, category_list)
                self.display_option_list.append(XmsDisplayMessage(
                    file=self.disp_opts_file,
                    edit_uuid=self.cov_uuid,
                ))
                break  # only one list
        return [], []

    def update_id_files(self):
        """Writes the display id files."""
        df = self.data.comp_ids.to_dataframe()
        df_bc_data = self.data.bc_data.to_dataframe()
        df_bc_data['id'] = df_bc_data['id'].astype(dtype='int')
        disp_names = BcData.display_list
        labels = BcData.display_labels
        for i in range(len(disp_names)):
            if type(disp_names[i]) is str:
                self._write_id_file(disp_names[i], df, df_bc_data, labels[i])
            else:
                for ix, name in enumerate(disp_names[i]):
                    self._write_id_file(name, df, df_bc_data, labels[i][ix])

    def _write_id_file(self, disp_name, df, df_bc_data, default_label):
        df1 = df.loc[df['display'] == disp_name]
        ids = df1['id'].astype(dtype='int').to_list()
        bc_ids = df1['bc_id'].astype(dtype='int').to_list()
        df2 = df_bc_data.loc[df_bc_data['id'].isin(bc_ids)]
        bc_id_to_json = {row.id: row.bc_json for row in df2.itertuples()}
        labels = [default_label] * len(ids)
        for idx, bc_id in enumerate(bc_ids):
            bc_json = bc_id_to_json.get(bc_id, '')
            json_dict = {} if not bc_json else json.loads(bc_json.encode())
            label = json_dict.get('label', '')
            if label:
                if 'upstream' in default_label:
                    label = label + ' up'
                elif 'downstream' in default_label:
                    label = label + ' down'
                labels[idx] = label

        path = os.path.join(os.path.dirname(self.main_file), 'display_ids')
        if not os.path.isdir(path):
            os.mkdir(path)
        id_file = os.path.join(path, f'{disp_name}.display_ids')
        write_display_option_ids(id_file, ids, labels)

    def open_arc_structure_plot(self, query, params, win_cont):
        """Opens the Assign Observation dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform..

        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid
        f_ids = params[0].get('selection', [])
        if len(f_ids) != 1:
            msg = 'One (and only one) arc must be selected to view Structure Output Plots.'
            return [('INFO', msg)], []

        data = PlotsAndTableDataSrh(
            feature_id=f_ids[0], feature_type='Arc', cov_uuid=self.data.cov_uuid, pe_tree=query.project_tree
        )
        dlg = PlotsAndTableDialog(win_cont, data)
        dlg.exec()

        return [], []

    def upgrade_hy8_from_version_1(self, query, params):
        """In version 1 each BC coverage had its own hy8 file.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists
        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid
        hy8_migrate = Hy8Migrate()
        return hy8_migrate.migrate(query)

    def check_external_hy8(self, query, params):
        """In version 1 each BC coverage had its own hy8 file, now there is 1 hy8 file for the project.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists
        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid
        # we only want to do this processing one time so we use a file in the temp area as the indicator
        tmp_dir = XmEnv.xms_environ_temp_directory()
        hy8_file = os.path.join(tmp_dir, 'check_external_hy8.txt')
        if not os.path.isfile(hy8_file):
            return [], []
        filesystem.removefile(hy8_file)

        comp_folder = os.path.join(tmp_dir, 'Components')
        proj_name = XmEnv.xms_environ_project_path()
        srh_folder = os.path.join(f'{os.path.splitext(proj_name)[0]}_models', 'SRH-2D')
        external_hy8 = os.path.join(srh_folder, 'culvert.hy8')
        internal_hy8 = os.path.join(comp_folder, 'shared_data', 'culvert.hy8')
        if not os.path.isfile(external_hy8):
            return [], []
        if filecmp.cmp(internal_hy8, external_hy8):
            return [], []
        mod_time = 0.0
        with open(internal_hy8, 'r') as f:
            lines = f.readlines()
            if lines and 'LAST_MOD_TIME' in lines[-1]:
                mod_time = float(lines[-1].split()[1])
        external_mod_time = os.path.getmtime(external_hy8)
        if mod_time > external_mod_time:
            return [], []

        msg = 'The project HY-8 file does not match the HY-8 file exported to SRH-2D.\n' \
              'For more details visit the link below:\n' \
              'https://www.xmswiki.com/wiki/SMS:SRH-2D_Structures#HY8_File'
        messages = [('INFO', msg)]
        return messages, []
