"""CoverageArcBuilder class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.data_objects.parameters import Arc, Point

# 4. Local modules


class CoverageArcBuilder:
    """Creates an arc for a coverage."""
    def __init__(self, node_locs, next_point_id=1, next_arc_id=1, existing_pts=None):
        """Constructor."""
        self._next_pt_id = next_point_id
        self._next_arc_id = next_arc_id
        self._node_locs = node_locs
        self._pt_map = existing_pts if existing_pts else {}
        self.arcs = []

    def _get_point(self, node_id):
        """Get an existing point or create a new one if it hasn't been added yet.

        Args:
            node_id (:obj:`int`): Node id from the geometry

        Returns:
            (:obj:`xms.data_objects.parameters.Point`): Coverage point associated with the passed in geometry node
        """
        if node_id not in self._pt_map:
            pt = Point(
                x=self._node_locs[node_id][0],
                y=self._node_locs[node_id][1],
                z=self._node_locs[node_id][2],
                feature_id=self._next_pt_id
            )
            self._next_pt_id += 1
            self._pt_map[node_id] = pt
        return self._pt_map[node_id]

    def add_arc(self, start_node_id, end_node_id, vert_list):
        """And an arc to the list of coverage arcs we are building up.

        Ensures we don't get duplicate points in our coverage.

        Args:
            start_node_id (:obj:`int`): Geometry node id of the arc's start node
            end_node_id (:obj:`int`): Geometry node id of the arc's end node
            vert_list (:obj:`list`): List of geometry node ids of the arc's vertices

        Returns:
            (:obj:`int`): The XMS feature arc id that was assigned to the new arc

        """
        vertices = [self._get_point(vert) for vert in vert_list]
        arc = Arc(
            feature_id=self._next_arc_id,
            start_node=self._get_point(start_node_id),
            end_node=self._get_point(end_node_id),
            vertices=vertices
        )
        self._next_arc_id += 1
        self.arcs.append(arc)
        return arc.id
