"""Imports SRH geom file."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import uuid

# 2. Third party modules
from PySide2.QtCore import QThread, Signal

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.components.display import windows_gui as win_gui
from xms.data_objects.parameters import UGrid
from xms.guipy.dialogs import xms_parent_dlg as xms_dlg
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg

# 4. Local modules
from xms.srh.file_io.geom_reader import GeomReader


class GeomImportRunner(QThread):
    """Worker thread for the .srhgeom import script."""
    processing_finished = Signal()

    def __init__(self):
        """Constructor."""
        super().__init__()
        self._logger = logging.getLogger('xms.srh')
        self._reader = None
        self._query = None
        self._sim_uuid = None

    def _add_xms_data(self):
        """Add imported data to Query Context to send back to SMS."""
        mesh = UGrid(
            self._reader.temp_mesh_file,
            name=self._reader.data['name'],
            uuid=str(uuid.uuid4()),
            projection=self._reader.geom_projection_from_grid_units()
        )

        # add the mesh to the Query Context
        self._query.add_ugrid(mesh)
        if self._sim_uuid:
            self._query.link_item(taker_uuid=self._sim_uuid, taken_uuid=mesh)

    def send(self):
        """Send imported data to SMS."""
        if self._query:
            self._query.send()

    def run(self):
        """Imports an SRH-2D mesh."""
        try:
            self._query = Query()
            self._sim_uuid = self._query.current_item_uuid()
            self._reader = GeomReader()
            self._reader.read(self._query.read_file)
            self._add_xms_data()
        except Exception:  # pragma no cover - hard to test exceptions using QueryPlayback
            self._logger.exception('Error importing SRH-2D mesh.')
        finally:
            self.processing_finished.emit()


def import_geom_with_feedback():
    """Import an .srhgeom file with the feedback dialog."""
    xms_dlg.ensure_qapplication_exists()

    # Parse parent HWNDs and icon path from commandline arguments.
    parent_hwnd, main_hwnd, _ = xms_dlg.parse_parent_window_command_args()
    win_cont = xms_dlg.get_parent_window_container(parent_hwnd)
    # Create the timer that keeps our Python dialog in the foreground of XMS.
    _ = win_gui.create_and_connect_raise_timer(main_hwnd, win_cont)  # Keep the timer in scope # noqa

    worker = GeomImportRunner()
    display_text = {
        'title': 'SRH-2D Mesh Import',
        'working_prompt': 'Importing SRH-2D mesh. Please wait...',
        'error_prompt': 'Error(s) encountered reading the SRH-2D mesh. Review log output for more details.',
        'success_prompt': 'Successfully imported SRH-2D mesh',
        'note': '',
        'auto_load': 'Auto load simulation into SMS when operation is complete',
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.srh', worker, win_cont)
    feedback_dlg.exec()
    win_gui.raise_main_xms_window(main_hwnd)  # Bring top-level Python window to foreground
    if not LogEchoQSignalStream.logged_error:  # Only send if no error.
        worker.send()  # Send data back to XMS
