"""MappedBcComponent class. Data mapped from a BC Coverage onto an SRH mesh."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
from PySide2.QtGui import QIcon

# 3. Aquaveo modules
from xms.components.display.display_options_io import read_display_options_from_json, write_display_options_to_json
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog
from xms.guipy.dialogs.xms_parent_dlg import get_xms_icon

# 4. Local modules
from xms.srh.components.mapped_material_component import MappedMaterialComponent
from xms.srh.components.srh_component import duplicate_display_opts


class MappedBcComponent(MappedMaterialComponent):
    """A hidden Dynamic Model Interface (DMI) component for the SRH-2D model simulation."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.

        """
        super().__init__(main_file)
        # [(menu_text, menu_method)...]
        self.tree_commands = [
            ('Display Options...', 'open_display_options'),
        ]
        self.disp_opts_file = os.path.join(os.path.dirname(self.main_file), 'bc_display_options.json')

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (:obj:`str`): Path to the new save location.
            save_type (:obj:`str`): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:

                new_main_file (:obj:`str`): Name of the new main file relative to new_path, or an absolute path
                if necessary.

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)

        if save_type == 'DUPLICATE':
            json_dict = duplicate_display_opts(new_path, 'bc_display_options.json')
            self.update_display_options(new_main_file, json_dict, action_requests)

        return new_main_file, messages, action_requests

    def open_display_options(self, query, params, win_cont):
        """Shows the display options dialog.

        Args:
            query (:obj:`xms.api.dmi.Query`)
            params (:obj:`list[str]`)
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        categories = CategoryDisplayOptionList()
        json_dict = read_display_options_from_json(self.disp_opts_file)
        categories.from_dict(json_dict)
        categories_list = [categories]

        dlg = CategoryDisplayOptionsDialog(categories_list, win_cont)
        icon_path = get_xms_icon()
        if icon_path:
            dlg.setWindowIcon(QIcon(icon_path))
        dlg.setModal(True)
        if dlg.exec():
            # write files
            category_lists = dlg.get_category_lists()
            for category_list in category_lists:
                write_display_options_to_json(self.disp_opts_file, category_list)
                self.display_option_list.append(
                    XmsDisplayMessage(file=self.disp_opts_file, draw_type=DrawType.draw_at_locations)
                )
                break  # only one list
        return [], []
