"""MonitorComponent class. Data for monitor Coverage."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import shutil

# 2. Third party modules
from PySide2.QtGui import QIcon

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest
from xms.components.display.display_options_io import (
    read_display_options_from_json, write_display_option_ids, write_display_options_to_json
)
from xms.components.display.xms_display_message import XmsDisplayMessage
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.plot_and_table_data_srh import PlotsAndTableDataSrh
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog
from xms.guipy.dialogs.plot_and_table_dialog import PlotsAndTableDialog
from xms.guipy.dialogs.xms_parent_dlg import get_xms_icon

# 4. Local modules
from xms.srh.components.srh_component import duplicate_display_opts
from xms.srh.components.srh_cov_component import SrhCoverageComponent
from xms.srh.data.monitor_data import MonitorData
from xms.srh.gui.monitor_dialog import run_dlg

MONITOR_PT_DISPLAY_OPTIONS_JSON = 'monitor_point_display_options.json'
MONITOR_ARC_DISPLAY_OPTIONS_JSON = 'monitor_arc_display_options.json'
MONITOR_PT_DISPLAY_ID_FILE = 'monitor.display_ids'
MEASURED_PT_DISPLAY_ID_FILE = 'measured_value.display_ids'
MONITOR_ARC_DISPLAY_ID_FILE = 'monitor_arc.display_ids'


class MonitorComponent(SrhCoverageComponent):
    """A hidden Dynamic Model Interface (DMI) component for the SRH-2D model simulation."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.

        """
        super().__init__(main_file)
        self.data = MonitorData(self.main_file)
        self.cov_uuid = self.data.info.attrs['cov_uuid']
        self.tree_commands = [
            ('Display Options...', 'open_display_options'),
        ]
        # [(menu_text, menu_method)...]
        self.point_commands = [
            ('Assign Point Properties...', 'open_assign_point_props'),
            ('Monitor Point Plots...', 'open_pt_monitor_plot'),
        ]
        self.arc_commands = [
            ('Assign Arc Properties...', 'open_assign_arc_props'),
            ('Monitor Line Plots...', 'open_arc_monitor_plot'),
        ]
        self.disp_opts_file = os.path.join(os.path.dirname(self.main_file), MONITOR_PT_DISPLAY_OPTIONS_JSON)
        self.arc_disp_opts_file = os.path.join(os.path.dirname(self.main_file), MONITOR_ARC_DISPLAY_OPTIONS_JSON)
        self._ensure_display_options_exist()

    def _ensure_display_options_exist(self):
        """Copy over default display options if this is the first time component has been instantiated."""
        if not self.main_file:
            return
        if not os.path.isfile(self.disp_opts_file):
            # Read the default display options, and save ourselves a copy with a randomized UUID.
            categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
            default_file = os.path.join(
                os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data',
                MONITOR_PT_DISPLAY_OPTIONS_JSON
            )
            json_dict = read_display_options_from_json(default_file)
            json_dict['comp_uuid'] = os.path.basename(os.path.dirname(self.main_file))
            categories.from_dict(json_dict)
            write_display_options_to_json(self.disp_opts_file, categories)
            # Save our display list UUID to the main file
            self.data.info.attrs['display_uuid'] = categories.uuid
            self.data.commit()
        else:
            categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
            json_dict = read_display_options_from_json(self.disp_opts_file)
            categories.from_dict(json_dict)
            if len(categories.categories) < 2:
                cats = CategoryDisplayOptionList()
                default_file = os.path.join(
                    os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data',
                    MONITOR_PT_DISPLAY_OPTIONS_JSON
                )
                json_dict = read_display_options_from_json(default_file)
                cats.from_dict(json_dict)
                categories.categories.append(cats.categories[1])
                write_display_options_to_json(self.disp_opts_file, categories)

        if not os.path.isfile(self.arc_disp_opts_file):
            # Read the default display options, and save ourselves a copy with a randomized UUID.
            categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
            default_file = os.path.join(
                os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data',
                MONITOR_ARC_DISPLAY_OPTIONS_JSON
            )
            json_dict = read_display_options_from_json(default_file)
            json_dict['comp_uuid'] = os.path.basename(os.path.dirname(self.main_file))
            categories.from_dict(json_dict)
            write_display_options_to_json(self.arc_disp_opts_file, categories)
            # Save our display list UUID to the main file
            self.data.info.attrs['arc_display_uuid'] = categories.uuid
            self.data.commit()

    def _update_point_id_files(self, new_comp_id):
        """Update the XMS component ids for pipe point categories whose lists have changed.

        Args:
            new_comp_id (:obj:`int`): The new component id

        """
        path = os.path.dirname(self.main_file)
        monitor_data = self.data.monitor_point_param_from_id(new_comp_id)
        if monitor_data.observation_point:
            updated_display_file = os.path.join(path, MEASURED_PT_DISPLAY_ID_FILE)
        else:
            updated_display_file = os.path.join(path, MONITOR_PT_DISPLAY_ID_FILE)
        fids, flabels = self.data.get_point_comp_ids_labels(monitor_data.observation_point)

        # Add the new component id to the target category.
        # updated_comp_ids = read_display_option_ids(updated_display_file)
        # updated_comp_ids.append(new_comp_id)
        write_display_option_ids(updated_display_file, fids, flabels)

        # Send back updated display lists to XMS after ActionRequest
        self.display_option_list.append(
            # Update the point display list only.
            XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid)
        )

    def _update_arc_id_files(self, new_comp_id):
        """Update the XMS component ids for pipe point categories whose lists have changed.

        Args:
            new_comp_id (:obj:`int`): The new component id

        """
        # Add the new component id to the target category.
        path = os.path.dirname(self.main_file)
        updated_display_file = os.path.join(path, MONITOR_ARC_DISPLAY_ID_FILE)

        fids, flabels = self.data.get_arc_comp_ids_labels()
        # updated_comp_ids = read_display_option_ids(updated_display_file)
        # updated_comp_ids.append(new_comp_id)
        write_display_option_ids(updated_display_file, fids, flabels)

        # Send back updated display lists to XMS after ActionRequest
        self.display_option_list.append(
            # Update the arc display list only.
            XmsDisplayMessage(file=self.arc_disp_opts_file, edit_uuid=self.cov_uuid)
        )

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (:obj:`bool`): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        self.data.load_all()
        self.data.commit()

        action = ActionRequest(
            modality='NO_DIALOG',
            class_name=self.class_name,
            module_name=self.module_name,
            main_file=self.main_file,
            method_name='get_initial_display_options'
        )
        return [], [action]

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (:obj:`str`): Path to the new save location.
            save_type (:obj:`str`): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:

                new_main_file (:obj:`str`): Name of the new main file relative to new_path, or an absolute path
                if necessary.

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)

        if save_type == 'DUPLICATE' or save_type == 'UNLOCK':
            _ = MonitorComponent(new_main_file)  # make sure display options exist
            json_dict = duplicate_display_opts(new_path, os.path.basename(self.disp_opts_file))
            arc_json_dict = duplicate_display_opts(new_path, os.path.basename(self.arc_disp_opts_file))
            data = MonitorData(new_main_file)
            data.load_all()
            if save_type == 'DUPLICATE':
                data.info.attrs['cov_uuid'] = ''
            data.info.attrs['display_uuid'] = json_dict['uuid']
            data.info.attrs['arc_display_uuid'] = arc_json_dict['uuid']
            data.commit()

        return new_main_file, messages, action_requests

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists
        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid

        self.display_option_list.extend(
            (
                XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid),
                XmsDisplayMessage(file=self.arc_disp_opts_file, edit_uuid=self.cov_uuid)
            )
        )
        return [], []

    def open_assign_point_props(self, query, params, win_cont):
        """Opens the Assign Observation dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid
        _kwargs = params[0]
        pt_ids = _kwargs.get('selection', [])
        if len(pt_ids) == 0:
            return [('INFO', 'No points selected. Select one or more points to assign properties.')], []

        # get the component ids
        id_files = _kwargs.get('id_files', [])
        if id_files is not None and len(id_files) > 1:
            self.load_coverage_component_id_map({'POINT': (id_files[0], id_files[1])})
        # Delete the id dumped by xms files.
        shutil.rmtree(os.path.join(os.path.dirname(self.main_file), 'temp'), ignore_errors=True)

        new_comp_id = run_dlg(pt_ids, self, win_cont)
        if new_comp_id is not None:
            self._update_point_id_files(new_comp_id)

        messages, actions = [], []
        return messages, actions

    def open_assign_arc_props(self, query, params, win_cont):
        """Opens the monitor arc dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid
        _kwargs = params[0]
        arc_ids = _kwargs.get('selection', [])
        if len(arc_ids) == 0:
            return [('INFO', 'No arcs selected. Select one or more arcs to assign properties.')], []

        # get the component ids
        id_files = _kwargs.get('id_files', [])
        if id_files is not None and len(id_files) > 1:
            self.load_coverage_component_id_map({'ARC': (id_files[0], id_files[1])})
        # Delete the id dumped by xms files.
        shutil.rmtree(os.path.join(os.path.dirname(self.main_file), 'temp'), ignore_errors=True)

        new_comp_id = run_dlg(arc_ids, self, win_cont, 'Arc')
        if new_comp_id is not None:
            self._update_arc_id_files(new_comp_id)

        messages, actions = [], []
        return messages, actions

    def open_display_options(self, query, params, win_cont):
        """Shows the display options dialog.

        Args:
            query (:obj:`xms.api.dmi.Query`):
            params (:obj:`list[str]`):
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid
        # point display opts
        categories = CategoryDisplayOptionList()
        json_dict = read_display_options_from_json(self.disp_opts_file)
        categories.from_dict(json_dict)
        categories_list = [categories]
        # arc display opts
        categories = CategoryDisplayOptionList()
        json_dict = read_display_options_from_json(self.arc_disp_opts_file)
        categories.from_dict(json_dict)
        categories_list.append(categories)

        dlg = CategoryDisplayOptionsDialog(categories_list, win_cont)
        icon_path = get_xms_icon()
        if icon_path:
            dlg.setWindowIcon(QIcon(icon_path))
        dlg.setModal(True)
        if dlg.exec():
            # write files
            category_lists = dlg.get_category_lists()
            disp_files = [self.disp_opts_file, self.arc_disp_opts_file]
            for category_list, disp_file in zip(category_lists, disp_files):
                write_display_options_to_json(disp_file, category_list)
            self.display_option_list.extend(
                (
                    XmsDisplayMessage(file=self.disp_opts_file, edit_uuid=self.cov_uuid),
                    XmsDisplayMessage(file=self.arc_disp_opts_file, edit_uuid=self.cov_uuid)
                )
            )
        return [], []

    def open_arc_monitor_plot(self, query, params, win_cont):
        """Opens the Assign Observation dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid
        f_ids = params[0].get('selection', [])
        feature_ids = f_ids if len(f_ids) > 1 else None
        data = PlotsAndTableDataSrh(
            feature_id=f_ids[0],
            feature_type='Arc',
            cov_uuid=self.data.cov_uuid,
            pe_tree=query.project_tree,
            model_name='SRH-2D',
            feature_ids=feature_ids
        )
        dlg = PlotsAndTableDialog(win_cont, data)
        dlg.exec()
        return [], []

    def open_pt_monitor_plot(self, query, params, win_cont):
        """Opens the Assign Observation dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid
        f_ids = params[0].get('selection', [])
        if len(f_ids) == 0:
            return [('INFO', 'No points selected. Select one or more points to view plots.')], []
        feature_ids = f_ids if len(f_ids) > 1 else None
        data = PlotsAndTableDataSrh(
            feature_id=f_ids[0],
            feature_type='Point',
            cov_uuid=self.data.cov_uuid,
            pe_tree=query.project_tree,
            model_name='SRH-2D',
            feature_ids=feature_ids
        )
        dlg = PlotsAndTableDialog(win_cont, data)
        dlg.exec()
        return [], []
