"""InitialConditionParameter class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt

# 3. Aquaveo modules
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate

# 4. Local modules
from xms.srh.components.parameters.parameter import Parameter


class InitialConditionParameter(Parameter):
    """Represents a parameter used in ParametersManager and the Advanced Simulation dialog."""
    def __init__(
        self, use=0, description='', default=0.0, string_value='', value=0.0, optimized_value=0.0, min=0.0, max=0.0
    ):
        """Initializes the class.

        Args:
            use (int): True if using this parameter
            description (str): Typically the arc number, or material name.
            default (float): default value
            string_value (str): string value
            value (float): value
            optimized_value (float): optimized value
            min (float): min
            max (float): max
        """
        super().__init__(
            use, -1, 'Initial condition', description, default, string_value, value, optimized_value, min, max
        )

    def id_string(self):
        """Returns a string for use in the 'id' column.

        Returns:
            (str): See description.
        """
        return 'init_cond'

    def add_run_data(self, disk_data, run_count, param_data):
        """Adds columns to the runs table.

        If disk_data has information about this parameter, uses it.

        Args:
            param_data (ParameterData): Param data.
            run_count (int): Number of runs.
            disk_data (dict): Param data from file.
        """
        self._add_column(disk_data, run_count, 'Initial Condition Type', object, param_data)
        self._add_column(disk_data, run_count, 'Initial Condition Value', object, param_data)

    def column_delegate(self, column_name, xy_series_names, dialog):
        """Give the name of a column in the runs table, returns the delegate, or None.

        Args:
            column_name (str): The name of the column in the runs table.
            xy_series_names (list): List of the names of the xy series.
            parent: Parent of the delegate.

        Returns:
            The delegate or None
        """
        delegate = None
        if column_name == 'Initial Condition Type':
            delegate = QxCbxDelegate(dialog)
            delegate.set_strings(
                [
                    'Dry', 'Initial Water Surface Elevation', 'Automatic', 'Restart File',
                    'Water Surface Elevation Dataset'
                ]
            )
        elif column_name == 'Initial Condition Value':
            from xms.srh.gui.qx_delegate_constant_file_dataset import QxDelegateConstantFileDataset  # circular depends
            delegate = QxDelegateConstantFileDataset(
                '',
                'Select Restart File',
                None,
                dialog.pe_tree_trimmed,
                dialog,
                query=dialog.query,
                allow_all_time_steps=False
            )
        return delegate

    def column_default(self, column_name):
        """Given a column name, return the default value.

        Args:
            column_name (str): The name of the column in the runs table.

        Returns:
            Default value.
        """
        if column_name == 'Initial Condition Type':
            return 'Dry'
        else:
            return ''

    def flags(self, index, column_name, flags):
        """Returns the flags for an item in the model.

        Args:
            index (QModelIndex): The item's model index.
            column_name (str): Name of the column.
            flags (Qt.ItemFlags): The flags in their initial state

        Returns:
            (Qt.ItemFlags): Updated flags for the item
        """
        if column_name.startswith('Initial Condition Value'):
            init_cond_type = get_init_cond_type(index)
            if init_cond_type != 'Initial Water Surface Elevation':
                flags = flags & (~Qt.ItemIsEditable)  # remove editable flag
        return flags


def get_init_cond_type(index):
    """Return the initial condition type given a QModelIndex at the initial condition value cell.

    Args:
        index (QModelIndex): Index of the cell.

    Returns:
        (str): Initial condition type (i.e. 'Dry', 'Initial Water Surface Elevation', 'Automatic', 'Restart File',
                                          'Water Surface Elevation Dataset')
    """
    model = index.model()
    init_cond_type = model.data(model.index(index.row(), index.column() - 1))
    return init_cond_type
