"""ManningsNParameter class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.srh.components.parameters.parameter import Parameter


class ManningsNParameter(Parameter):
    """Represents a parameter used in ParametersManager and the Advanced Simulation dialog."""
    def __init__(
        self,
        use=0,
        id_number=-1,
        description='',
        default=0.0,
        string_value='',
        value=0.0,
        optimized_value=0.0,
        min=0.0,
        max=0.0
    ):
        """Initializes the class.

        Args:
            use (int): True if using this parameter
            id_number (int): Number used in the id_string.
            description (str): Typically the arc number, or material name.
            default (float): default value
            string_value (str): string value
            value (float): value
            optimized_value (float): optimized value
            min (float): min
            max (float): max
        """
        super().__init__(
            use, id_number, "Manning's N", description, default, string_value, value, optimized_value, min, max
        )

    def id_string(self):
        """Returns a string for use in the 'id' column.

        Returns:
            (str): See description.
        """
        return f'mat_{self.id_number}'

    def add_run_data(self, disk_data, run_count, param_data):
        """Adds columns to the runs table.

        If disk_data has information about this parameter, uses it.

        Args:
            param_data (ParameterData): Param data.
            run_count (int): Number of runs.
            disk_data (dict): Param data from file.
        """
        self._add_column(disk_data, run_count, self.column_base_name(), 'float', param_data)

    def calibratable(self):
        """Returns True if the parameter can be used for calibration.

        Returns:
            (bool): See description.
        """
        return True
