"""SimComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import json
import os
from pathlib import Path

# 2. Third party modules
import orjson
from PySide2.QtWidgets import QDialog, QWidget

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Query, XmsEnvironment as XmEnv
from xms.api.tree import tree_util
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg
from xms.guipy.param import param_h5_io

# 4. Local modules
from xms.srh.components import generate_floodway_evaluation_lines_runner as gfelr
from xms.srh.components.coverage_mapper_runner import CoverageMapperRunner
from xms.srh.components.parameters.parameters_manager import ParametersManager
from xms.srh.components.srh_component import SrhComponent
from xms.srh.components.statistician_runner import run_statistics_with_feedback
from xms.srh.data.model_control import ModelControl
from xms.srh.data.par import par_util
from xms.srh.data.par.data_io_string import DataIoStringParam
from xms.srh.data.par.summary_report_dialog_data import SummaryReportDialogData
from xms.srh.file_io.report import plots
from xms.srh.file_io.report.structure_coverage_looper import StructureCoverageLooper
from xms.srh.file_io.report.summary_report_generator import SummaryReportGenerator
from xms.srh.floodway.create_floodway_simulation import create_floodway_simulation_with_feedback
from xms.srh.gui.compare_simulation_data_dialog import CompareSimulationDataDialog  # noqa: I202
from xms.srh.gui.generate_floodway_evaluation_lines_dialog import GenerateFloodwayEvaluationLinesDialog
from xms.srh.gui.generate_floodway_simulation_dialog import GenerateFloodwaySimulationDialog
from xms.srh.gui.model_control_dialog import ModelControlDialog
from xms.srh.gui.simulation_plots_dialog import SimulationPlotsDialog
from xms.srh.gui.structures_map_dialog import StructuresMapDialog
from xms.srh.gui.summary_report_dialog import SummaryReportDialog


class SimComponent(SrhComponent):
    """A hidden Dynamic Model Interface (DMI) component for the SRH-2D model simulation."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.

        """
        super().__init__(main_file)
        self.data = ModelControl()
        if os.path.isfile(self.main_file):
            self.data.output.output_maximum_datasets = False  # for compatibility with old datasets
            tmp_data = DataIoStringParam()
            param_h5_io.read_from_h5_file(self.main_file, tmp_data)
            if tmp_data.text:
                pd = json.loads(tmp_data.text)
                par_util.orjson_dict_to_param_cls(pdict=pd, cls=self.data, df_handler=None)
            else:  # old file version
                param_h5_io.read_from_h5_file(self.main_file, self.data)
        else:
            self._write_data_to_main_file()
            # param_h5_io.write_to_h5_file(self.main_file, self.data)
        #                    [(menu_text, menu_method)...]
        self.tree_commands = [
            ('Model Control...', 'open_model_control'),
            ('Generate Snap Preview', 'create_snap_preview'),
        ]
        self.tool_commands = [
            ('Advanced Simulation (Beta)...', 'advanced_simulation'),
            ('View Simulation Plots', 'simulation_plots'),
            ('Statistical Analysis (Beta)', 'statistics'),
            ('Summary Report (Beta)...', 'summary_report'),
            ('View Structures Map...', 'structures_map'),
            # ('Datasets To Rasters (Beta)...', 'datasets_to_rasters'),
            ('Generate Floodway Simulation (Beta)', 'generate_floodway_simulation'),
            ('Generate Hydraulic Evaluation Lines (Beta)...', 'generate_floodway_evaluation_lines'),
            ('Compare Simulation Data (Beta)...', 'compare_simulation_data'),
        ]

    def save_sim_data_to_main_file(self):
        """Saves the current simulation data to the main file."""
        self._write_data_to_main_file()

    def _write_data_to_main_file(self):
        """Writes this component's data to its main file."""
        self.data.param.file_type.precedence = 100
        self.data.param.file_version.precedence = 101
        self.data.param.mesh_uuid.precedence = 102
        self.data.advanced.param.raster_uuid.precedence *= -1
        tmp_data = DataIoStringParam()
        pd = par_util.param_cls_to_orjson_dict(cls=self.data, df_handler=None, skip_negative_precedence=True)
        tmp_data.text = json.dumps(pd)
        param_h5_io.write_to_h5_file(self.main_file, tmp_data)

    def link_event(self, link_dict, lock_state):
        """This will be called when one or more coverages, ugrids, or other components are linked to this component.

        Args:
            link_dict (:obj:`dict`): A dictionary with keys being UUIDs as strings representing the objects being
                linked into this component. The values of this dictionary are a list of strings of the parameter names
                of the "takes" from the XML that this is a part of.
            lock_state (:obj:`bool`): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        messages = []
        actions = []
        for link_uuid, link_xml_params in link_dict.items():
            for xml_param in link_xml_params:
                if xml_param in ['mesh', 'ugrid']:
                    if self.data.mesh_uuid and self.data.mesh_uuid != link_uuid:
                        # Already have a linked 2DMesh or UGrid
                        params = {'old_uuid': self.data.mesh_uuid}
                        action = ActionRequest(
                            main_file=self.main_file,
                            modality='NO_DIALOG',
                            class_name=self.class_name,
                            module_name=self.module_name,
                            method_name='delete_old_domain',
                            comp_uuid=self.uuid,
                            parameters=params
                        )
                        actions.append(action)
                    self.data.mesh_uuid = link_uuid
        self._write_data_to_main_file()
        return messages, actions

    def unlink_event(self, unlinks, lock_state):
        """This will be called when a coverage, or a ugrid, or another component is unlinked from this component.

        Args:
            unlinks (:obj:`list[str]`): A list of UUIDs as strings representing the objects being unlinked.
            lock_state (:obj:`bool`): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        if self.data.mesh_uuid in unlinks:
            self.data.mesh_uuid = ''
        self._write_data_to_main_file()
        return [], []

    def delete_old_domain(self, query, params):
        """Delete the existing linked domain when another is linked to the simulation.

        This is needed because we want to allow using an SMS 2DMesh or UGrid object as the domain. The xml has both a
        take_mesh2d and take_ugrid parameter with a limit of 1. If we get a link event for the domain and we already
        have one, we need to remove the old because it is of a different type

        Args:
            query (:obj:`xms.api.dmi.Query.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists

        """
        old_uuid = None
        sim_uuid = None
        if params and params[0]:
            old_uuid = params[0].get('old_uuid')
            # Need the UUID of the parent tree item, even though it is the hidden component that actually has the XML
            # take parameter.
            sim_uuid = query.parent_item_uuid()
        if old_uuid and sim_uuid:
            query.unlink_item(taker_uuid=sim_uuid, taken_uuid=old_uuid)
        return [], []

    def open_model_control(self, query, params, win_cont):
        """Opens the Model Control dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        dlg = ModelControlDialog(win_cont, self, query.copy_project_tree(), query)
        if dlg.exec():
            self.data = dlg.data
            self._write_data_to_main_file()
        return [], []

    def create_snap_preview(self, query, params, win_cont):
        """Creates mapped components to display SRH data on a mesh.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        note = ''
        worker = CoverageMapperRunner(query)
        error_str = 'Error(s) encountered applying coverages to simulation. Review log output for more details.'
        warning_str = 'Warning(s) encountered applying coverages to simulation. Review log output for more details.'
        display_text = {
            'title': 'SRH-2D Snap Preview',
            'working_prompt': 'Applying coverages to mesh. Please wait...',
            'error_prompt': error_str,
            'warning_prompt': warning_str,
            'success_prompt': 'Successfully created snap preview',
            'note': note,
            'auto_load': 'Close this dialog automatically when exporting is finished.'
        }
        feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.srh', worker, win_cont)
        if feedback_dlg.exec():  # Don't send data if user canceled.
            worker.query_helper.add_mapped_components_to_xms()
        return [], []

    def generate_floodway_simulation(self, query, params, win_cont):
        """Generates a floodway simulation.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        dlg = GenerateFloodwaySimulationDialog(win_cont, query.copy_project_tree())
        if dlg.exec() == QDialog.Accepted:
            wse_maximum_rise, mat_uuid = dlg.get_inputs()
            create_floodway_simulation_with_feedback(query, win_cont, self, wse_maximum_rise, mat_uuid)
        return [], []

    def generate_floodway_evaluation_lines(self, query, params, win_cont):
        """Generates floodway evaluation lines.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        dlg = GenerateFloodwayEvaluationLinesDialog(parent=win_cont, query=query)
        if dlg.exec():
            gfelr.run_generate_floodway_evaluation_lines_with_feedback(win_cont, dlg.generator)
        return [], []

    def compare_simulation_data(self, query, params, win_cont):
        """Compares floodway data.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        dlg = CompareSimulationDataDialog(parent=win_cont, query=query)
        dlg.exec()
        return [], []

    def advanced_simulation(self, query, params, win_cont):
        """Opens the Parameters dialog.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        parameters_manager = ParametersManager(self.main_file, self.data)
        parameters_manager.run_parameters_dialog(query=query, win_cont=win_cont)
        return [], []

    def statistics(self, query, params, win_cont):
        """Opens the Statistics dialog.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with SMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        run_statistics_with_feedback(query, win_cont)
        return [], []

    def _report_options_filename(self):
        """Returns the filepath to the summary report options json file.

        Returns:
            See description.
        """
        return os.path.join(os.path.dirname(self.main_file), 'summary_report.json')

    def _save_report_dialog_data(self, report_param):
        """Saves the summary report options in the param class to disk.

        Args:
            report_param (:obj:`SummaryReportDialogData`): Param class holding the data.
        """
        pdict = par_util.param_cls_to_orjson_dict(report_param, None)
        report_options_filename = self._report_options_filename()
        with open(report_options_filename, 'wb') as file:
            file.write(orjson.dumps(pdict))

    def _load_report_dialog_data(self):
        """Reads the summary report options from disk and returns a SummaryReport class.

        If there is no file, a default class is returned.

        Returns:
            See description.
        """
        report_param = SummaryReportDialogData()
        try:
            report_options_filename = self._report_options_filename()
            with open(report_options_filename, 'rb') as file:
                pdict = orjson.loads(file.read())
                par_util.orjson_dict_to_param_cls(pdict, report_param, None)
        except:  # noqa
            pass
        return report_param

    def summary_report(self, query, params, win_cont):
        """Creates the project summary report.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with SMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        # Check that the project has been saved.
        if not query.xms_project_path:
            return [('INFO', 'The project must be saved before generating a Summary Report.')], []

        report_dialog_data = self._load_report_dialog_data()
        dlg = SummaryReportDialog(win_cont, report_dialog_data)
        testing = os.path.isfile('c:\\temp\\debug_xms_srh_summary_report.dbg')
        if testing or dlg.exec() == QDialog.Accepted:
            self._save_report_dialog_data(report_dialog_data)
            query.xms_agent.set_timeout(90000)  # Large scatter can take awhile. Set to 90 seconds.
            report_generator = SummaryReportGenerator(query, report_dialog_data, testing)
            filename = report_generator.run(win_cont, True)  # 3rd arg is True for feedback. Use False to debug.
            if filename:
                os.startfile(filename, 'open')  # Open file in default app

        return [], []

    def structures_map(self, query: Query, params: list, win_cont: QWidget):
        """Creates the project summary report.

        Args:
            query: Object for communicating with SMS
            params: Generic map of parameters. Unused in this case.
            win_cont: The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        # Check that the project has been saved.
        if not query.xms_project_path:
            return [('INFO', 'The project must be saved before viewing structures plot.')], []

        # Get structures data
        temp_dir = XmEnv.xms_environ_process_temp_directory()
        coverage_looper = StructureCoverageLooper(None, temp_dir, query, None)
        structures_list = coverage_looper.visit_coverages()
        if not structures_list:
            return [('INFO', 'No structures found in the project.')], []

        # Create plot and open dialog
        projection = query.display_projection
        plot_file = plots.plot_structure(structures_list, projection, temp_dir)
        plot_file = Path(temp_dir) / plot_file
        dlg = StructuresMapDialog(win_cont, plot_file)
        if dlg.exec() == QDialog.Accepted:
            pass
        return [], []

    def simulation_plots(self, query, params, win_cont):
        """Creates images of the plots from the simulation run.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with SMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        # Check that the project has been saved.
        if not query.xms_project_path:
            return [('INFO', 'The project must be saved before viewing simulation plots.')], []

        # get the tree item name
        sim_uuid = query.parent_item_uuid()
        sim_item = tree_util.find_tree_node_by_uuid(query.project_tree, sim_uuid)
        tree_item_name = sim_item.name

        # get the path to the SRH output files
        proj_file_name = os.environ.get('XMS_PYTHON_APP_PROJECT_PATH')
        proj_name = os.path.splitext(os.path.basename(proj_file_name))[0]
        proj_path = os.path.dirname(proj_file_name)
        sim_dir = os.path.join(proj_path, f'{proj_name}_models', 'SRH-2D', tree_item_name)
        if not os.path.isdir(sim_dir):
            sim_dir = os.path.join(proj_path, f'{proj_name}', 'SRH-2D', tree_item_name)
        inf_files = []
        param_data = ParametersManager.read_parameter_file(self.main_file)
        if param_data and param_data['use_parameters'] and param_data['run_type'] == 'Scenarios':
            run_count = param_data['run_count']
            for run in range(run_count):
                run_name = param_data['runs']['Run'][run]
                run_dir = os.path.join(sim_dir, run_name)
                inf_files.append(os.path.join(run_dir, f'{run_name}_INF.dat'))
        else:
            case_name = f'{self.data.hydro.case_name}_INF.dat'
            inf_files.append(os.path.join(sim_dir, case_name))

        for inf_file in inf_files:
            if not os.path.isfile(inf_file):
                msg = f'No SRH-2D solution found. Run the simulation to view plots. Missing file: {inf_file}.'
                return [('INFO', msg)], []

        dlg = SimulationPlotsDialog(win_cont, inf_files, self.data.enable_sediment)
        if dlg.exec() == QDialog.Accepted:
            pass

        return [], []
