"""BcDataInternalSink class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import pandas as pd
import param

# 3. Aquaveo modules
from xms.guipy.param.param_enabler import ParamEnabler  # noqa: I202

# 4. Local modules


class BcDataInternalSink(param.Parameterized):
    """Bc type internal-ss data."""
    sink_flow_type = param.ObjectSelector(
        label='Sink flow (Q) type',
        default='Constant',
        objects=['Constant', 'Time series', 'Weir', 'Rating curve'],
        doc='option for specifying the flow type',
        precedence=1,
    )

    constant_q = param.Number(
        label='Constant Q (outflow is positive)',
        default=0.0,
        doc='constant flow',
        precedence=2,
    )
    constant_q_units = param.ObjectSelector(
        label='',
        default='cfs',
        objects=['cfs', 'cms'],
        doc='units for flow',
        precedence=3,
    )

    time_series_q = param.DataFrame(
        default=pd.DataFrame(columns=['hrs', 'vol_per_sec']),
        doc='time series for Q',
        precedence=4,
    )
    time_series_q_units = param.ObjectSelector(
        label='',
        default='hrs -vs- cfs',
        objects=['hrs -vs- cfs', 'hrs -vs- cms'],
        doc='units for flow',
        precedence=5,
    )

    weir_coeff = param.Number(
        label='Coeff (Cw)',
        default=0.0,
        doc='weir coefficient',
        precedence=6,
    )
    weir_crest_elevation = param.Number(
        label='Crest elevation (Zb)',
        default=0.0,
        doc='weir crest elevation',
        precedence=7,
    )
    weir_length_across = param.Number(
        label='Length across weir (Lw)',
        default=0.0,
        doc='Length across weir',
        precedence=8,
    )
    weir_units = param.ObjectSelector(
        default='Feet',
        objects=['Feet', 'Meters'],
        doc='units for weir',
        precedence=9,
    )
    weir_use_total_head = param.Boolean(
        label='Use total head',
        default=False,
        doc='option to use total head',
        precedence=10,
    )

    rating_curve = param.DataFrame(
        default=pd.DataFrame(columns=['vol_per_sec', 'WSE']),
        doc='series for rating curve',
        precedence=11,
    )
    rating_curve_units = param.ObjectSelector(
        label='',
        default='cfs -vs- feet',
        objects=['cfs -vs- feet', 'cms -vs- meters'],
        doc='units for rating curve',
        precedence=12,
    )

    def __init__(self):
        """Initializes the class."""
        super().__init__()
        self.enabler = ParamEnabler(
            param_cls=self,
            enablers=['sink_flow_type'],
            depends={
                'Constant': ['constant_q', 'constant_q_units'],
                'Time series': ['time_series_q', 'time_series_q_units'],
                'Weir':
                    ['weir_coeff', 'weir_crest_elevation', 'weir_length_across', 'weir_units', 'weir_use_total_head'],
                'Rating curve': ['rating_curve', 'rating_curve_units']
            },
        )

    @param.depends('sink_flow_type', watch=True)
    def _update_sink_flow_type(self):
        """Watches changes made to the sink_flow_type object."""
        self.enabler.do_enabling(force_disable=False)
