"""Creates a Boundary Conditions coverage hidden component."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.display.display_options_io import write_display_option_ids
from xms.core.filesystem import filesystem
from xms.data_objects.parameters import Component

# 4. Local modules
from xms.srh.components import bc_component as bcc
from xms.srh.data.bc_data import BcData
from xms.srh.file_io.hy8_migrate import Hy8Migrate


def build_bc_component(bcs, bc_cov_uuid, comp_dir, att_ids, hy8_file, new_comp_uuid=None):
    """Creates the BC coverage's hidden component.

    This method is for building a BC coverage from scratch.

    Args:
        bcs (:obj:`dict`): {bc_id: BcDataParam}. Both arcs of structures should appear in bcs with different bc_id but
            pointing to the same BcDataParam
        bc_cov_uuid (:obj:`str`): UUID of the BC coverage the new component belongs to.
        comp_dir (:obj:`str`): Path to the XMS temp components folder (where the component will be created).
        att_ids (:obj:`list`): List of the feature arc att ids to assign component ids to. att_ids must be in the same
            order as the keys in bcs (list needs to follow insertion order of the dict).
        hy8_file (:obj:`str`): filename of hy8 input file
        new_comp_uuid (:obj:`str`): UUID of the new BC component. Useful for testing, usually just want
            randomly generated.

    Returns:
        (:obj:`xms.data_objects.parameters.Component`): data_object for the new BC component
    """
    # Create a new UUID and folder for the component data
    if new_comp_uuid:  # Testing, use hard-coded UUID
        comp_uuid = new_comp_uuid
    else:
        comp_uuid = str(uuid.uuid4())  # pragma: no cover
    bc_comp_dir = os.path.join(comp_dir, comp_uuid)
    os.makedirs(bc_comp_dir, exist_ok=True)
    os.makedirs(os.path.join(bc_comp_dir, 'display_ids'), exist_ok=True)
    bc_main_file = os.path.join(bc_comp_dir, 'bc_comp.nc')

    # Create the data_object Component to send back to SMS
    bc_do_comp = Component(main_file=bc_main_file, comp_uuid=comp_uuid, model_name='SRH-2D', unique_name='Bc_Component')

    # Load component data with stuff we read in
    bc_comp = bcc.BcComponent(bc_main_file)
    comp_data = bc_comp.data
    comp_data.info.attrs['cov_uuid'] = bc_cov_uuid
    # Set coverage UUID member on component so we can load component ids without committing data.
    disp_dict = {'Upstream': 0, 'Downstream': 1, 'Inflow monitor': 2, 'Outflow monitor': 3}
    used_bc_ids = set()
    next_comp_id = 1
    bc_comp.cov_uuid = comp_data.info.attrs['cov_uuid']
    comp_ids = {bc_id: 0 for bc_id in bcs.keys()}
    for bc_id, data in bcs.items():
        disp_category = BcData.display_list[data.bc_type_index()]
        is_structure = data.bc_type in BcData.structures_list
        if not is_structure:
            # Since we are importing from a native file, we can initialize component ids to just be the
            # arc ids if not a structure.
            comp_data.set_bc_id_display_with_comp_id(bc_id, disp_category, next_comp_id)
            comp_ids[bc_id] = next_comp_id
            next_comp_id += 1
        else:
            if bc_id not in used_bc_ids:
                comp_ids[data.arcs.arc_id_0] = next_comp_id
                comp_ids[data.arcs.arc_id_1] = next_comp_id + 1
                used_bc_ids.add(data.arcs.arc_id_0)
                used_bc_ids.add(data.arcs.arc_id_1)
                disp_0 = disp_category[disp_dict[data.arcs.arc_option_0]]
                comp_data.set_bc_id_display_with_comp_id(bc_id, disp_0, next_comp_id)
                disp_1 = disp_category[disp_dict[data.arcs.arc_option_1]]
                comp_data.set_bc_id_display_with_comp_id(bc_id, disp_1, next_comp_id + 1)

                # make a structure
                data.arcs.arc_id_0 = next_comp_id
                data.arcs.arc_id_1 = next_comp_id + 1
                next_comp_id += 2
                comp_data.set_structure_with_bc_id(data.arcs, bc_id)
                comp_data.set_structure_with_bc_id(data.arcs, bc_id)  # do this twice for structures
        comp_data.set_bc_data_with_id(data, bc_id)
    bc_comp.update_id_files()
    comp_data.commit()

    # copy hy8 file if it exists
    if os.path.isfile(hy8_file) and os.path.getsize(hy8_file) > 0:
        # comp_hy8_file = os.path.join(os.path.dirname(bc_main_file), 'culvert.hy8')
        shared_folder = os.path.join(comp_dir, 'shared_data')
        if not os.path.isdir(shared_folder):
            os.mkdir(shared_folder)
        comp_hy8_file = os.path.join(shared_folder, 'culvert.hy8')
        if not os.path.isfile(comp_hy8_file) or os.path.getsize(comp_hy8_file) == 0:
            filesystem.copyfile(hy8_file, comp_hy8_file)
        else:
            migrate = Hy8Migrate()
            migrate.merge_file_into_project(proj_hy8=comp_hy8_file, bc_data=comp_data, bc_hy8=hy8_file)

    # Write component id and BC arc att ids to a file so we can initialize them in get_initial_display_options
    id_file = os.path.join(bc_comp_dir, bcc.BC_INITIAL_ATT_ID_FILE)
    write_display_option_ids(id_file, att_ids)
    id_file = os.path.join(bc_comp_dir, bcc.BC_INITIAL_COMP_ID_FILE)
    ids = list(comp_ids.values())
    write_display_option_ids(id_file, ids)

    return bc_do_comp
