"""Writes the *.srhgeom files."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class GeomWriter:
    """Writes SRH-2D geom file."""
    def __init__(self, file_name, ugrid, grid_name, grid_units, node_strings):
        """Constructor.

        Args:
            file_name (:obj:`str`): Path to the output file
            ugrid (:obj:`xms.grid.ugrid.UGrid`): The SRH-2D domain geometry
            grid_name (:obj:`str`): Name of the SRH-2D mesh
            grid_units (:obj:`str`): Horizontal units of the geometry
            node_strings (:obj:`list[list]`): List of the boundary condition nodestrings.
                [[bc1_id, [bc1_pt1, ..., bc1_ptN]], ..., [bcN_id, [bcN_pt1, ..., bcN_ptN]]]
        """
        self._file_name = file_name
        self._ugrid = ugrid
        self._grid_name = grid_name
        self._grid_units = grid_units
        self._node_strings = node_strings

    def write(self):
        """Write the geometry file."""
        with open(self._file_name, 'w') as file:
            file.write('SRHGEOM 30\n')
            file.write(f'Name "{self._grid_name}"\n\n')
            file.write(f'{self._grid_units}\n')
            cell_stream = self._ugrid.cellstream
            cell_id = 0
            i = 0
            while i < len(cell_stream):
                cell_id += 1
                # cell_type = cell_stream[i]
                num_pts = cell_stream[i + 1]
                line = f'Elem {cell_id}'
                for j in range(num_pts):
                    line += f' {cell_stream[i + 2 + j] + 1}'
                file.write(f'{line}\n')
                i = i + 2 + num_pts
                # file.write(f'Elem {cell_id} {cell_stream[i+2] + 1} {cell_stream[i+3] + 1} {cell_stream[i+4] + 1}')
                # if num_pts == 4:
                #     file.write(f' {cell_stream[i+5] + 1}\n')
                #     i = i + 6
                # else:
                #     file.write('\n')
                #     i = i + 5
            pts = self._ugrid.locations
            for index in range(len(pts)):
                file.write(f'Node {index + 1} {pts[index][0]} {pts[index][1]} {pts[index][2]}\n')

            if self._node_strings:
                for i in range(len(self._node_strings)):
                    file.write(f'NodeString {self._node_strings[i][0]}')
                    nodes = self._node_strings[i][1]
                    for j in range(len(nodes)):
                        file.write(f' {nodes[j] + 1}')
                        if (j + 1) % 10 == 0:
                            file.write('\n')
                    if len(nodes) % 10 != 0:
                        file.write('\n')
