"""Write sediment parameters to the *.srhhydro file."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class HydroSedimentWriter:
    """Writes SRH-2D geom file."""
    def __init__(self, file, model_control, logger):
        """Constructor.

        Args:
            file: The output file stream
            model_control (:obj:`ModelControl`): The simulation data
            logger (:obj:`logging.Logger`): Logger to use
        """
        self._file = file
        self._model_control = model_control
        self.logger = logger

    def write(self):
        """Write the sediment model control parameters."""
        mc = self._model_control
        if not mc.enable_sediment:
            return
        self._file.write(f'GenSedParams {mc.sediment.sediment_specific_gravity}')
        diameters = mc.sediment.particle_diameter_threshold['Particle Diameter Threshold (mm)'].tolist()
        diameters.sort()
        for d in diameters:
            self._file.write(f' {d}')
        self._file.write('\n')
        bed_opt = mc.sediment.shear_partitioning_option
        if bed_opt != 'Percentage':
            bed_opt = 'Scaled_D90'
        d90 = mc.sediment.shear_partitioning_scaled_d90_factor
        percent = mc.sediment.shear_partitioning_percentage
        self._file.write(f'BedShearOption {bed_opt} {d90} {percent}\n')
        if mc.sediment.specify_sediment_simulation_start_time:
            self._file.write(f'SedimentStartTime {mc.sediment.sediment_simulation_start_time}\n')
        self._write_sediment_equation(self._file, mc.sediment.transport_equation_parameters, '')
        self._write_sediment_parameters(self._file, mc.sediment.transport_parameters)
        self._write_sediment_cohesion(self._file, mc.sediment)

    def _write_sediment_equation(self, file, sed_eq, card_append_str):
        """Write the sediment parameters to the hydro file.

        Args:
            file: The output hydro file stream
            sed_eq (:obj:`SedimentEquationParameters`): Sediment equation parameters class
            card_append_str (:obj:`str`): string to append to the file cards when writing the mixed 'lower' and
                'higher' equation parameters
        """
        eq_dict = {
            'Engelund-Hansen': 'EH',
            'Meyer-Peter-Muller': 'MPM',
            'Parker': 'PARKER',
            'Wilcock-Crowe': 'WILCOCK',
            'Wu-et-al': 'WU',
            'Yang 1973 Sand w 1984 Gravel': 'YANG73',
            'Yang 1979 Sand w 1984 Gravel': 'YANG79',
            'Mixed': 'MIXED'
        }
        if not card_append_str:
            file.write(f'SedEquation {eq_dict[sed_eq.sediment_transport_equation]}\n')
        if sed_eq.sediment_transport_equation == 'Meyer-Peter-Muller':
            file.write(f'CapacityHidingFactor{card_append_str} {sed_eq.meyer_peter_muller_hiding_factor}\n')
        elif sed_eq.sediment_transport_equation == 'Parker':
            file.write(
                f'SedCapacityCoeff{card_append_str} {sed_eq.parker_reference_shields_parameter}'
                f' {sed_eq.parker_hiding_coefficient}\n'
            )
        elif sed_eq.sediment_transport_equation == 'Wilcock-Crowe':
            file.write(
                f'WilcockCoeff{card_append_str} {sed_eq.wilcox_t1_coefficient} {sed_eq.wilcox_t2_coefficient}'
                f' {sed_eq.wilcox_sand_diameter}\n'
            )
        elif sed_eq.sediment_transport_equation == 'Wu-et-al':
            file.write(f'CritShieldsParam{card_append_str} {sed_eq.wu_critical_shields_parameter}\n')
        elif sed_eq.sediment_transport_equation == 'Mixed':
            lower = sed_eq.lower_transport_parameters
            higher = sed_eq.higher_transport_parameters
            file.write(
                f'MixedSedParams {int(sed_eq.mixed_sediment_size_class_cutoff)}'
                f' {eq_dict[lower.sediment_transport_equation]}'
                f' {eq_dict[higher.sediment_transport_equation]}\n'
            )
            self._write_sediment_equation(file, lower, 'Lower')
            self._write_sediment_equation(file, higher, 'Higher')

    def _write_sediment_parameters(self, file, sed_par):
        """Write the sediment parameters to the hydro file.

        Args:
            file: The output hydro file stream
            sed_par (:obj:`SedimentParameters`): Sediment equation parameters class
        """
        file.write(f'WaterTemperature {sed_par.water_temperature}\n')
        file.write(f'SusLoadAdaptationCoeffs {sed_par.deposition_coefficient} {sed_par.erosion_coefficient}\n')

        all_items = sed_par.param.adaptation_length_bedload_mode.objects
        bed_opt = all_items.index(sed_par.adaptation_length_bedload_mode)
        file.write(f'BedLoadAdaptationLength {bed_opt}')
        if bed_opt > 1:
            msg = f'Adaptation Length for Bedload is using a deprecated option for Mode: {all_items[bed_opt]}. ' \
                  f'Change to a non-deprecated option in the Sediment tab of the Model Control dialog.'
            self.logger.error(msg)
        if sed_par.adaptation_length_bedload_mode == 'Constant Length':
            file.write(f' {sed_par.adaptation_length_bedload_length}')
        file.write('\n')

        act_lay_opt = 1
        act_lay_val = sed_par.active_layer_constant_thickness
        if sed_par.active_layer_thickness_mode == 'Thickness based on D90':
            act_lay_opt = 2
            act_lay_val = sed_par.active_layer_d90_scale
        file.write(f'ActiveLayerThickness {act_lay_opt} {act_lay_val}\n')

    def _write_sediment_cohesion(self, file, sed):
        """Write the sediment parameters to the hydro file.

        Args:
            file: The output hydro file stream
            sed (:obj:`ModelControlSediment`): Sediment parameters class
        """
        file.write('CohesiveSediment ')
        if not sed.enable_cohesive_sediment_modeling:
            file.write('0\n')
            return
        file.write('1\n')

        unit_str = 'EN' if sed.cohesive.bulk_density_units == 'English' else 'SI'
        file.write(f'CohesiveProperties {sed.cohesive.bulk_density} {unit_str}\n')
        fv_opt = -1
        if sed.cohesive.fall_velocity == 'Severn River Properties':
            fv_opt = 0
        elif sed.cohesive.fall_velocity == 'Velocity Data File (mm/sec)':
            # the filename should be on the "FallVelocity" line, not a separate line
            # fv_opt = 1
            fv_opt = f'"{sed.cohesive.fall_velocity_data_file}"'
            if not sed.cohesive.fall_velocity_data_file:
                fv_opt = -1
                msg = 'Fall Velocity option is set to "Velocity Data File (mm/sec)" but no data file is set. ' \
                      'Set a data file in the Sediment tab of the Model Control dialog.'
                self.logger.error(msg)
        file.write(f'FallVelocity {fv_opt}\n')
        # if sed.cohesive.fall_velocity == 'Velocity Data File (mm/sec)':
        #     file.write(f'FallVelocityFile "{sed.cohesive.fall_velocity_data_file}"\n')

        er_opt = 0
        if sed.cohesive.erosion_rate == 'Erosion Data File':
            er_opt = f'"{sed.cohesive.erosion_rate_data_file}"'
        file.write(f'CohesiveErosionRate {er_opt}\n')
        if sed.cohesive.erosion_rate != 'Erosion Data File':
            unit_str = 'SI' if sed.cohesive.erosion_units == 'Metric' else 'EN'
            file.write(
                f'CohesiveErosionParams {sed.cohesive.surface_erosion} {sed.cohesive.mass_erosion}'
                f' {sed.cohesive.surface_erosion_constant} {sed.cohesive.mass_erosion_constant}'
                f' {unit_str}\n'
            )

        unit_str = 'SI' if sed.cohesive.deposition_units == 'Metric' else 'EN'
        file.write(
            f'CohesiveDepositionParams {sed.cohesive.full_depostion} {sed.cohesive.partial_depostion}'
            f' {sed.cohesive.equilibrium_concentration} {unit_str}\n'
        )
